proc(y,mu) = genglm(code,x,b,opt)
; -----------------------------------------------------------------
; Library       glm
; -----------------------------------------------------------------
;  See_also     glmest genmultlo
; -----------------------------------------------------------------
;   Macro       genglm
; -----------------------------------------------------------------
;   Description  genglm generates data from a GLM model.
; -----------------------------------------------------------------
;   Link  ../tutorials/glmstart.html Tutorial: GLM in XploRe 
; -----------------------------------------------------------------
;   Usage        {y,mu} = genglm(code, x, b {,opt})
;   Input
;     Parameter   code  
;     Definition        text string, the short code for the model (e.g. 
;                       "bilo" for logit or "noid" for ordinary PLM).
;     Parameter   x  
;     Definition        n x k  matrix, design.
;     Parameter   b  
;     Definition        k x 1  vector, coefficients.
;     Parameter   opt
;     Definition        optional, a list with optional input. The macro
;                       "glmopt" can be used to set up this parameter.
;                       The order of the list elements is not important.
;                       Parameters which are not given are replaced by 
;                       defaults (see below).
;     Parameter   opt.phi
;     Definition        nuisance parameter, usually the dispersion 
;                       parameter. 
;     Parameter   opt.wx
;     Definition        scalar or n x 1 vector, the binomial index 
;                       vector, only considered for binomial models. 
;                       If not given, set to 1.
;     Parameter   opt.off
;     Definition        scalar or n x 1 vector, offset. 
;                       If not given, set to 0.
;     Parameter   opt.pow
;     Definition        optional, power for power link.
;     Parameter   opt.nbk
;     Definition        scalar, extra parameter k for negative  
;                       binomial distribution. 1/k is supposed
;                       to be a positive integer. 
;                       If not given, set to 1 (geometric distribution).
;   Output
;     Parameter   y
;     Definition        n x 1  vector, response.
;     Parameter   mu
;     Definition        n x 1  vector, regression.
; -----------------------------------------------------------------
;   Example   library("glm")
;             n = 100
;             b = 2|(-1) 
;             x = normal(n,rows(b))
;             opt = glmopt("wx",round(uniform(n).*5)+1)
;             {y,mu} = genglm("bilo",x,b,opt)             
; -----------------------------------------------------------------
;   Result    y[i]'s are pseudo-random variables with distribution 
;             Binomial(wx[i],mu[i]) where mu[i]=1/(1+exp(-x[,i]*b).
; -----------------------------------------------------------------
;   Notes     The Gamma models "gacl" and "gapow" are
;             not yet implemented.
; -----------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/18
; -----------------------------------------------------------------
;
; classify our model
;
  glmmodels=getglobal("glmmodels")
;
  binomial  = sum(code==(glmmodels.binomial)) >0
  gaussian  = sum(code==(glmmodels.gaussian)) >0
  poisson   = sum(code==(glmmodels.poisson)) >0
  gamma     = sum(code==(glmmodels.gamma))>0
  igaussian = sum(code==(glmmodels.igaussian))>0
  nbinomial = sum(code==(glmmodels.nbinomial))>0
;
  error(rows(dim(x))>2,"x must be vector or matrix")
;
  pow=0
  nbk=1
  wx=1
  off=0
  phi=1
;
  n = rows(x)
  p = cols(x)
  b = vec(b)
  error(rows(b)!=p,"dimensions of x and b are incompatible!")
;
  if (exist(opt)>0)
    if (comp(opt,"wx")>0)
      notgood=(dim(dim(opt.wx))!=1)*(rows(opt.wx)!=n)+(rows(dim(opt.wx))!=1)
      notgood=notgood+(sum(abs(wx))==0)+(min(wx)<0)
      error(notgood>0, "weights in opt.wx are bad!")
      if (notgood==0)
        wx=opt.wx
      endif
    endif
    if (comp(opt,"off")>0)
      notgood=(dim(dim(opt.off))!=1)*(rows(opt.off)!=n)+(rows(dim(opt.off))!=1)
      error(notgood>0, "offsets in opt.off are bad!")
      if (notgood==0)
        off=opt.off
      endif
    endif
    if (comp(opt,"pow")>0)
      notgood=(dim(dim(opt.pow))!=1)
      warning (notgood>0, "opt.pow was unusable, used default =0 instead")
      if (notgood==0)
        pow=opt.pow
      endif
    endif
    if (comp(opt,"nbk")>0)
      notgood=(dim(dim(opt.nbk))!=1)
      warning (notgood>0, "opt.nbk was unusable, used default =1 instead")
      if (notgood==0)
        nbk=opt.nbk
      endif
    endif
    if (comp(opt,"phi")>0)
      notgood=(dim(dim(opt.phi))!=1)
      warning (notgood>0, "opt.phi was unusable, used default =1 instead")
      if (notgood==0)
        phi=opt.phi
      endif
    endif
  endif
;
  eta=x*b+off
  mu=glmlink(code,eta,list(nbk,pow))
;
  switch
    case (binomial)
      if (code=="bipro")
        wx1=max(wx)
        wx0=min(wx)
        u=normal(n,wx1)
        i=((1:wx1)' <= wx)
        y=sum((eta+u>0).*i,2)
      else
        wx1=max(wx)
        wx0=min(wx)
        u=uniform(n,wx1)
        i=((1:wx1)' <= wx.*matrix(n))
        y=sum(((u<mu).*i),2)
      endif
      break
    case (gaussian)
      u=normal(n).*sqrt(phi)
      y=mu+u
      break
    case (poisson)
      ; this is a quick & dirty implementation of
      ; Devroyes Lemma 3.3 (page 504), 
      ; a little bit memory intensive - but it works ...
      m1=ceil(2*max(mu))
      em=exp(-mu)
      u=uniform(n,m1)
      tmp=min(prod(u,2)<em)
      while (tmp==0)
        u=u~uniform(n,m1)
        tmp=min(prod(u,2)<em)
      endo
      tmp=cumprod(u,2)<exp(-mu)
      y=maxind(cumprod(u,2)<exp(-mu),2)-1
      break
    case (gamma)
      error(1,"Gamma distribution is not yet implemented!")
      break
    case (igaussian)
      ; implementation of Devroyes Lemma 4.1 (pages 148-149) 
      u=normal(n)
      u=u^2
      y1=mu+mu^2.*u.*(phi*0.5) 
      y1=y1-sqrt(mu.*u./(4*phi)+(mu^2).*(u^2)).*mu.*(phi*0.5)
      tmp=uniform(n)<=(mu/(mu+y1))
      y=y1.*(tmp)+((mu^2)./y1).*(1-tmp)
      break
    case (nbinomial)
      ; implementation acc. to Devroye (page 543) and 
      ; Gentle (page 101). Hopefully ok...
      nu=1/nbk
      bad=(ceil(nu)!=(nu))
      error(bad,"only implemented if (1./nbk) is an integer !")
      u=uniform(n,nu)
      p=mu/(mu+nu)
      y=sum(floor(log(u)./log(p)),2)
      break
    default;
      error(1,"model code '"+code+"' is unknown!")
      break
  endsw
;
endp




