proc(m)=gintest(code,t,y,h,g,loc,opt)   
; -----------------------------------------------------------------
;   Library  	  gam
; -----------------------------------------------------------------
;   See_also      intest, intestpl, gintestpl, gamfit, pcad,
; -----------------------------------------------------------------
;   Macro         gintest
; -----------------------------------------------------------------
;   Description   estimation of the univariate additive functions
;                 in a separable generalized additive model 
;                 using Nad.Watson, local linear or local quadratic
; -----------------------------------------------------------------
;   Usage         m = gintest(code,t,y,h,g,loc{,opt})    
;   Input
;     Parameter   code  
;     Definition      string, specifying the code function 
;                     implemented codes: noid, bipro, bilo
;     Parameter   t 
;     Definition      n x p  matrix, the continuous predictor variables.
;     Parameter   y 
;     Definition      n x q matrix , the observed 
;                     response variables
;     Parameter   h  
;     Definition      p x 1 or 1 x 1 matrix , chosen bandwidth for 
;                     the directions of interest         
;     Parameter   g  
;     Definition      p x 1 or 1 x 1 matrix , chosen bandwidth for 
;                     the directions not of interest       
;     Parameter   loc  
;     Definition      dummy , for loc=0 local constant (Nad. Wats.),
;                     for loc=1 local linear and for loc=2 local
;                     quadratic estimator will be used
;     Parameter   opt
;     Definition      optional, a list with optional input. The macro
;                     "gplmopt" can be used to set up this parameter.
;                     The order of the list elements is not important.
;                     Parameters which are not given are replaced by 
;                     defaults (see below).
;     Parameter   opt.tg  
;     Definition      ng x pg vector, a grid for continuous part. If tg is
;                     given, the nonparametric function will also be 
;                     computed on this grid.
;     Parameter   opt.shf  
;     Definition      integer,  (show-how-far) if exists and =1, an output
;                     is produced which indicates how the iteration
;                     is going on (additive function / point of estimation /
;                     number of iteration).
;   Output                        
;     Parameter   m 
;     Definition      n(ng) x p(pg) x q matrix, containing the marginal
;                     integration estimators
; -----------------------------------------------------------------
;   Example   library("gam")
;             randomize(1235)
;             n     = 100
;             p     = 2
;             t     = uniform(n,p)*2-1
;             g1    = 2*t[,1]
;             g2    = t[,2]^2
;             g2    = g2 - mean(g2)
;             m     = g1 + g2
;             y     = cdfn(m) .> uniform(n)    ; probit model
;             h     = #(1.7, 1.5)		
;             g     = #(1.7, 1.5)
;             tg    = grid(-0.8,0.1,19)
;             opt   = gamopt("tg",tg~tg,"shf",1)
;             loc   = 1
;             code  = "bipro"
;             m     = gintest(code,t,y,h,g,loc,opt)
;             d1    = tg[,1]~m[,1]
;             d2    = tg[,2]~m[,2]
;             setmaskp(d1,4,4,8)
;             setmaskp(d2,4,4,8)
;             bild  = createdisplay(1,2)
;             show(bild,1,1,d1,t[,1]~g1)
;             show(bild,1,2,d2,t[,2]~g2)
; -----------------------------------------------------------------
;   Result    the marginal integration estimator of the additive 
;             functions, see Linton and Haerdle (1996)
; -----------------------------------------------------------------
;   Author    Sperlich & Stockmeyer  970711
; -----------------------------------------------------------------
 if (comp("opt","tg")>0)
   tg = opt.tg
 else
   tg = t
 endif
 if (comp("opt","shf")>0)
   shf = opt.shf
 else
   shf = 0
 endif
 p = cols(t)
 n = rows(t)
 q = cols(y)
 pg = cols(tg)
 ng = rows(tg)
 o = sort(t[,1]~(1:n))[,2]
 t = t[o] 
 y = y[o]
 error(sum(loc==#(0,1,2))!=1,"loc is not correctly specified")
 if (rows(g)<>p)             /* reshape g to matrix */
   errtext = "rows of g didn't fit: "+string("%1.4f",p)+" assumed"
   warning(rows(g)>1,errtext)
   g = matrix(p)*g[1,1]  
 endif 
 if (rows(h)<>pg)            /* reshape h to matrix */
   errtext = "rows of h didn't fit: "+string("%1.4f",pg)+" assumed"
   warning(rows(h)>1,errtext)
   h = matrix(pg)*h[1,1]
 endif
 j = 1
 if (loc==0)
   m = matrix(ng,pg,q)
   y = y~matrix(n)               
   while (j<=pg)
      i = 1
      hs = g
      hs[j] = h[j]
      while (i<=ng)
      if (shf)
         ("gintest: "+string("%1.0f",j)+" "+string("%1.0f",i))
      endif
        test = t
        test[,j] = tg[i,j]*matrix(n)
        r = sker(t,hs',"qua",y,test)
        r = paf(r,r[,q+1].<>0)
        mh = r[,1:q]./r[,q+1]
        switch
          case (code=="noid")
            m[i,j,] = mean(mh,1)
            break
          case (code=="bipro")
            mh = mh.*(mh.>0)+(mh.<=0)*0.00001
            mh = mh.*(mh.<1)+(mh.>=1)*0.99999
            m[i,j,] = mean(qfn(mh),1)
            break
          case (code=="bilo")
            mh = mh.*(mh.>0)+(mh.<=0)*0.00001
            mh = mh.*(mh.<1)+(mh.>=1)*0.99999
            m[i,j,] = mean(log(mh/(1-mh)),1)
            break
          default
            error(1,"code function not yet implemented")
        endsw  
        i = i + 1
      endo   
      j = j + 1
   endo
   y = y[,1:q]
 else
   loc = loc-1               /* still needed that way for locpoldis */
   m = matrix(ng,(2+loc)*pg,q)*0
   while (j.<=pg)
     hs = g
     hs[j] = h[j]
     if (loc)
       posm = j | (pg+j) | (2*pg+j)
       posmb = 1 | (1+j) | (1+p+j)
     else
       posm = j | (pg+j)
       posmb = 1 | (1+j)
     endif
     i = 1 
     while (i.<=ng)
       if (shf)
         ("gintest: "+string("%1.0f",j)+" "+string("%1.0f",i))
       endif
       test = t
       test[,j] = tg[i,j]*matrix(n)
       ma = locpoldis(t,test,y,hs,loc)
       k = 1
       while (k.<=q)
         mb = ma[,,k]
         sel = isNumber(mb[,1])  
         if (sum(sel,1)>10)
           mb = paf(mb,sel)
         endif
         switch
           case (code=="noid")
             m[i,posm,k] = mean(mb[,posmb],1)
             break
           case (code=="bipro")
             mb = mb.*(mb.>0)+(mb.<=0)*0.00001
             mb = mb.*(mb.<1)+(mb.>=1)*0.99999
             m[i,posm,k] = mean(qfn(mb[,posmb]),1)
             break
           case (code=="bilo")
             mb = mb.*(mb.>0)+(mb.<=0)*0.00001
             mb = mb.*(mb.<1)+(mb.>=1)*0.99999
             m[i,posm,k] = mean(log(mb[,posmb]/(1-mb[,posmb])),1)
             break
           default
             error(1,"code function not yet implemented")
         endsw  
         k = k + 1
       endo
       i = i + 1
     endo   
   sel  = isNumber(m[,j,1]) 
   j = j+1
   endo
   if (loc)
     m[,(2*pg+1):(3*pg),] = m[,(2*pg+1):(3*pg),]*2 
   endif
 endif
endp
