proc(xr,yr,wxr,offr,ctrl)=glminit(code,x,y,opt) 
; ----------------------------------------------------------------------------
; Library       glm
; ----------------------------------------------------------------------------
;  See_also     glmopt glmest glmcore glmstat
; ----------------------------------------------------------------------------
;   Macro       glminit
; ----------------------------------------------------------------------------
;   Description  glminit checks the validity of input and performs the 
;                initial calculations for an GLM fit.
;                The output is ready to be used with glmcore.
; ----------------------------------------------------------------------------
;   Link  ../tutorials/glmstart.html Tutorial: GLM in XploRe 
; ----------------------------------------------------------------------------
;   Usage        {xr,yr,wxr,offr,ctrl} = glminit(code,x,y{,opt})
;   Input
;     Parameter   code  
;     Definition        text string, the short code for the model (e.g. 
;                       "bilo" for logit or "noid" for ordinary PLM).
;     Parameter   x  
;     Definition        n x p  matrix, the predictor variables.
;     Parameter   y  
;     Definition        n x 1  vector, the response variables. Binomial
;                       y[i] may have (integer) values between 0 
;                       and opt.wx[i] or opt.wx (if opt.wx is scalar). 
;     Parameter   opt
;     Definition        optional, a list with optional input. The macro
;                       "glmopt" can be used to set up this parameter.
;                       The order of the list elements is not important.
;                       Parameters which are not given are replaced by 
;                       defaults (see below).
;     Parameter   opt.weights
;     Definition        string, type of weights. Can be "frequency"
;                       for replication counts, or "prior" (default)
;                       for prior weights in weighted regression.
;     Parameter   opt.wx
;     Definition        scalar or n x 1 vector, frequency or prior 
;                       weights. If not given, set to 1.
;     Parameter   opt.off
;     Definition        scalar or n x 1 vector, offset. Can be used for
;                       constrained estimation. If not given, set to 0.
;     Parameter   opt.shf  
;     Definition        integer, if exists and =1, some output is produced 
;                       which indicates how the iteration is going on.
;     Parameter   opt.miter
;     Definition        integer, maximal number of iterations. The default 
;                       is 10.
;     Parameter   opt.cnv
;     Definition        integer, convergence criterion. The default is 0.0001.
;     Parameter   opt.fscor
;     Definition        integer, if exists and =1, a Fisher scoring is
;                       performed (instead of the default Newton-Raphson
;                       procedure). This parameter is ignored for
;                       canonical links.
;     Parameter   opt.pow
;     Definition        optional, power for power link.
;     Parameter   opt.nbk
;     Definition        scalar, extra parameter k for negative binomial
;                       distribution. If not given, set to 1 (geometric 
;                       distribution).
;   Output
;     Parameter   xr
;     Definition        nr x p  matrix, the predictor variables,
;                       eventually reduced up to replications.
;     Parameter   yr
;     Definition        nr x 1  or nr x 2  or nr x 3 matrix, 
;                       either the response values, or sums of response 
;                       values in 1st column and sums of a function
;                       of response values in the 2nd and 3rd column (e.g.
;                       sums of y^2 or log(y), see glmll).
;                       (In the case of replicated data, the number of 
;                       replications should be given in wx, yr[,1] contains
;                       the sums of all responses for a replication, 
;                       yr[,2:3] contains sums of e.g. y^2 or log(y) for a 
;                       replication.)
;     Parameter   wxr
;     Definition        nr x 1 vector or scalar, weights. 
;     Parameter   offr
;     Definition        nr x 1 vector or scalar, offset. 
;     Parameter   ctrl  
;     Definition        6 x 1 integer vector or scalar, contains control 
;                       parameters shf, miter, cnv, fscor, pow, nbk
;                       or  shf alone. 
;                       Defaults for miter, cnv, fscor, pow, nbk are 
;                       10, 0.0001, 0,  0 (logarithm link) and 1. 
; ----------------------------------------------------------------------------
;   Example   library("glm")
;             ;==========================
;             ;  simulate data 
;             ;==========================
;             n=100
;             b=1|2
;             p=rows(b)
;             x=2.*uniform(n,p)-1
;             y=( 1./(1+exp(-x*b)).>uniform(n) )
;             ;==========================
;             ;  GLM fit 
;             ;==========================
;             opt=glmopt("shf",1,"norepl",1)
;             {x,y,wx,off,ctrl}=glminit("bilo",x,y,opt)
;             lf=glmcore("bilo",x,y,wx,off,ctrl)
;             b~lf.b
; ----------------------------------------------------------------------------
;   Result    A logit fit for E[y|x,t] is computed. The options "shf" for
;             information during the iteration and "norepl" for not to
;             search for replications are set. lf.b contains the 
;             coefficients for the linear part. The example gives the 
;             true b together with the GLM estimate lf.b.
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/18
; ----------------------------------------------------------------------------
;
; classify our algo
;
  glmmodels=getglobal("glmmodels")
;
  binomial  = sum(code==(glmmodels.binomial)) >0
  gaussian  = sum(code==(glmmodels.gaussian)) >0
  poisson   = sum(code==(glmmodels.poisson)) >0
  gamma     = sum(code==(glmmodels.gamma))>0
  igaussian = sum(code==(glmmodels.igaussian))>0
  nbinomial = sum(code==(glmmodels.nbinomial))>0
;
  canonical = sum(code==(glmmodels.canonical)) >0
  direct    = sum(code==(glmmodels.direct)) >0
  power     = sum(code==(glmmodels.power)) >0
;
; check initial errors
;
  bad=(rows(dim(x))>2)
  error(bad>0,"x must be vector or matrix")
  error(rows(x)!=rows(y),"x and y have different number of rows")
  error((rows(dim(y))>1),"y must be a vector")
;
  n=rows(x)
  p=cols(x)
  one=matrix(n)
;
; set the defaults
;
  wx=1
  off=0
  shf=0
  norepl=0
  miter=10
  cnv=0.0001
  fscor=0
  pow=0
  nbk=1
  weights="prior"
;
; now check which optional values have been given
;
  if (exist(opt)>0)
    if (comp(opt,"wx")>0)
      notgood=(dim(dim(opt.wx))!=1)*(rows(opt.wx)!=n)+(rows(dim(opt.wx))!=1)
      notgood=notgood+(sum(abs(wx))==0)+(min(wx)<0)
      error(notgood>0, "weights in opt.wx are bad!")
      if (notgood==0)
        wx=opt.wx
      endif
    endif
    if (comp(opt,"weights")>0)
      notgood=(exist(opt.weights)!=2)
      error(notgood>0, "wrong type of weights in opt.weights!")
      if (notgood==0)
        weights=opt.weights
      endif
    endif
    if (comp(opt,"off")>0)
      notgood=(dim(dim(opt.off))!=1)*(rows(opt.off)!=n)+(rows(dim(opt.off))!=1)
      error(notgood>0, "offsets in opt.off are bad!")
      if (notgood==0)
        off=opt.off
      endif
    endif
    if (comp(opt,"shf")>0)
      shf=(opt.shf==1)*(dim(dim(opt.shf))==1)
    endif
    if (comp(opt,"fscor")>0)
      fscor=(opt.fscor==1)*(dim(dim(opt.fscor))==1)
    endif
    if (comp(opt,"norepl")>0)
      norepl=(opt.norepl==1)*(dim(dim(opt.norepl))==1)
    endif
    if (comp(opt,"miter")>0)
      tmp=floor(opt.miter[1,1])
      notgood=(tmp<1)+(dim(dim(opt.miter))!=1)
      warning (notgood>0, "opt.miter was unusable, used default =10 instead")
      miter=notgood*opt.miter+(1-notgood)*tmp
    endif
    if (comp(opt,"pow")>0)
      notgood=(dim(dim(opt.pow))!=1)
      warning (notgood>0, "opt.pow was unusable, used default =0 instead")
      if (notgood==0)
        pow=opt.pow
      endif
    endif
    if (comp(opt,"nbk")>0)
      notgood=(dim(dim(opt.nbk))!=1)
      warning (notgood>0, "opt.nbk was unusable, used default =1 instead")
      if (notgood==0)
        nbk=opt.nbk
      endif
    endif
    if (comp(opt,"cnv")>0)
      tmp=opt.cnv[1,1]
      notgood=(tmp<=0)+(dim(dim(tmp))!=1)
      warning (notgood>0, "opt.cnv was unusable, used default =0.0001 instead")
      cnv=notgood*cnv+(1-notgood)*tmp
    endif
  endif
;
; check y with respect to model
;
  switch
    case (binomial)
      error(min(y) <0 ,"gplm/code='"+code+"': y has negative values")
      error(max(y)==0 ,"gplm/code='"+code+"': y has wrong values")
      if (weights=="prior")
        if (max(paf(y,wx.*one))>1)
          error(sum(y>wx)!=0,"gplm/code='"+code+"': y has wrong values")
          y = y./(wx+(wx==0)) ; divide by wx, but not by 0!
        endif
        y = y~((y+0.5)./2) ; ((y.*wx+0.5)./(wx+1)) 
      else
        y = y~((y+0.5)./2)
      endif
      break
    case (poisson)
      error(min(y)<0 ,"glm/code='"+code+"': y has negative values")
      break
    case (gamma)
      error(min(y)<=0,"glm/code='"+code+"': y has non-positive values")
      break
    case (igaussian)
      error(min(y)<=0,"glm/code='"+code+"': y has non-positive values")
      break
    case (nbinomial)
      error(min(y)<0 ,"glm/code='"+code+"': y has negative values")
      break
    default
      ; all is allowed ...
  endsw
;
; search for replications
;
  if (norepl!=1)
    switch
      case (gaussian)
        y=y~(y^2)
        break
      case (poisson || gamma)
        y=y~log(y+(y==0))
        break
      case (igaussian)
        y=y~((y!=0)./(y+(y==0)))
        break
      case (nbinomial)
        y=y~log(y+(y==0))~log(y+1/nbk)
        break
      default ; (binomial)
        ; nothing to do
    endsw
;    if (weights!="prior")
;      {xr,yr}=discrete(x~y~(off.*one),(wx.*one))
;      offr=xr[,cols(xr)]
;      wxr =yr[,2]
;      yr  =xr[,(p+1):(cols(xr)-1)]
;      xr  =xr[,1:p]
;    else
      {xr,yr}=discrete(x~(off.*one),(wx.*one)~y)
      offr=xr[,cols(xr)]
      xr  =xr[,1:p]
      wxr =yr[,2]
      yr  =yr[,3:cols(yr)]./yr[,1]
;    endif 
;
;wx'
;  if (weights=="prior")
;    wx=(wx.*n)./sum(wx.*one)  ; normalize weights to sum to n
;  endif
;wx'
;sum(wx)
    maxoff=max(offr)
    maxwx =max(wxr)
    if (maxoff==min(offr))
      offr=maxoff
    endif
    if (maxwx==min(wxr))
      wxr=maxwx
    endif
  else
    xr=x
    yr=y
    wxr=wx
    offr=off
  endif
;
; check xr 
;
  error (countNaN(inv(xr'*(wxr.*xr)))>0, "matrix x is singular!!")  
;
; define control parameter vector
;
  ctrl   =0.*matrix(6)
  ctrl[1]=shf
  ctrl[2]=miter
  ctrl[3]=cnv
  ctrl[4]=fscor*(1-canonical)
  ctrl[5]=pow
  ctrl[6]=nbk
endp 




