proc(b,bv,mu,stat)=glmmultlo(x,y,opt) 
; ----------------------------------------------------------------------------
; Library       glm
; ----------------------------------------------------------------------------
;  See_also     glmest genmultlo
; ----------------------------------------------------------------------------
;   Macro       glmmultlo
; ----------------------------------------------------------------------------
;   Description  glmmultlo fits a  multinomial/conditional logit model
;                where the response Y is multinomial distributed. 
;                This means, P( Y = j | Xa , Xi) is proportional to
;                exp( Xa * ba + Xi * bi[j] ). Here,  Xi  denotes that
;                part of the explanatory variables which merely depends
;                on the individuals and  Xa  covers variables which may 
;                vary with the alternatives  j. Either part, Xa or Xi, 
;                can be omitted.
; ----------------------------------------------------------------------------
;   Link  ../tutorials/glmstart.html Tutorial: GLM in XploRe 
; ----------------------------------------------------------------------------
;   Usage        myfit = glmmultlo(x,y{,opt})
;   Input
;     Parameter   x  
;     Definition        n x r matrix, the predictor variables. 
;                       Individual-specific variables form single 
;                       columns of x. Alternative-specific variables
;                       must be evaluated for each alternative and 
;                       are to be stored in blocks of m subsequent
;                       columns. The optional parameters opt.indiv
;                       and opt.alter define which columns belong
;                       to which group. Without these optional
;                       parameters, all columns of x are interpreted
;                       as individual-specific.
;     Parameter   y  
;     Definition        n x 1 vector or n x m matrix. If y is vector
;                       (numeric or string), the different
;                       realizations are considered to be the
;                       alternatives. If y is matrix, it should
;                       contain 0/1 dummies, with 1 in the j-th column
;                       indicating that alternative j (in 1 ... m) has
;                       been chosen. 
;     Parameter   opt
;     Definition        optional, a list with optional input. The macro
;                       "glmopt" can be used to set up this parameter.
;                       The order of the list elements is not important.
;                       Parameters which are not given are replaced by 
;                       defaults (see below).
;     Parameter   opt.shf  
;     Definition        integer, if exists and =1, some output is produced 
;                       which indicates how the iteration is going on.
;     Parameter   opt.miter
;     Definition        integer, maximal number of iterations. The default 
;                       is 10.
;     Parameter   opt.cnv
;     Definition        scalar, convergence criterion. The default is 0.0001.
;     Parameter   opt.indiv
;     Definition        indices of columns of x (=variables) which are
;                       individual-specific, i.e. which are constant
;                       over alternatives. If neither opt.indiv and
;                       opt.alter are given, the model is estimated as
;                       if opt.indiv=1:r. Otherwise, the default is
;                       empty.
;     Parameter   opt.alter
;     Definition        indices of columns of x (=variables) which are
;                       alternative-specific, i.e. which vary over 
;                       the alternatives (and eventually individuals). 
;                       x[,opt.alter] should have a multiple of m 
;                       columns, i.e. contain subsequent realizations 
;                       for each of the alternatives. The default is empty.  
;   Output
;     Parameter   myfit
;     Definition        list with the components b, bv, and stat:
;     Parameter   b.indiv
;     Definition        p x m vector, estimated coefficients
;                       corresponding to individual-specific variables. 
;                       The first column is zero.
;     Parameter   b.alter
;     Definition        q x 1 vector, estimated coefficients corresponding
;                       to alternative-specific variables.
;     Parameter   bv.indiv
;     Definition        (p*m) x (p*m) matrix, estimated covariance 
;                       for b.indiv, matrix of p x p blocks.      
;     Parameter   bv.alter
;     Definition        q x q  matrix, estimated covariance for b.alter.
;     Parameter   bv.mixed
;     Definition        (p*m) x q matrix, estimated mixed covariances 
;                       between b.alter and b.indiv, vector of p x q 
;                       blocks.      
;     Parameter   mu
;     Definition        n x m vector, estimated response mu=P(y=j)
;                       corresponding to alternatives 1 ... m of y.
;     Parameter   stat
;     Definition        list with the following statistics:
;     Parameter   stat.serror
;     Definition        standard errors, list containing components
;                       indiv and/or alter, respectively.
;     Parameter   stat.tvalues
;     Definition        t-values, list containing components
;                       indiv and/or alter, respectively.
;     Parameter   stat.pvalues
;     Definition        p-values, list containing components
;                       indiv and/or alter, respectively.
;     Parameter   stat.deviance
;     Definition        deviance,
;     Parameter   stat.loglik
;     Definition        log-likelihood,
;     Parameter   stat.r2
;     Definition        (pseudo) R^2.
;     Parameter   stat.adr2
;     Definition        adjusted (pseudo) R^2.
;     Parameter   stat.it
;     Definition        scalar, number of iterations needed  
;     Parameter   stat.ret
;     Definition        scalar, return code: 
;                         0  o.k., 
;                         1  maximal number of iterations reached,
;                        -1  missing values have been encountered.
; ----------------------------------------------------------------------------
;   Example   library("glm") 
;             n = 100
;             b = list((0|0)~(1|2)~(-1|0)~(1|1),2)
;             m = cols(b{1})
;             id=diag(matrix(m))
;             x=normal(n,2)~kron(id,matrix(n/m))
;             {y,mu}  = genmultlo(x,b) 
;             ;
;             opt=glmopt("indiv",1:2,"alter",3:6)
;             ml = glmmultlo(x,y,opt)
;             ml.b
;             ml.stat
; ----------------------------------------------------------------------------
;   Result    Multinomial logit data are generated. The first
;             two columns of the design 'x' correspond to 
;             individual-specific influences, the other four
;             columns are alternative-specific.
;             Then, a multinomial logit fit is computed. 
;             The optional parameter 'opt' comprises the 
;             information how to split the design.
;             'ml.b' gives the coefficients, 'ml.stat' some
;             statistics of the fit.
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/18
; ----------------------------------------------------------------------------
;
  bad=(rows(dim(x))>2)
  error(bad>0,"design must be vector or matrix")
  error(rows(x)!=rows(y),"design and response are incomaptible!")
  error((rows(dim(y))>2),"response must be vector or matrix")
;
  n=rows(x)
  p=cols(x)
;
  wx=1
  off=0
  shf=0
  norepl=0
  miter=10
  cnv=0.0001
;
; now check which optional values have been given
;
  if (exist(opt)>0)
    if (comp(opt,"wx")>0)
      notgood=(dim(dim(opt.wx))!=1)*(rows(opt.wx)!=n)+(rows(dim(opt.wx))!=1)
      notgood=notgood+(sum(abs(wx))==0)+(min(wx)<0)
      error(notgood>0, "weights in wx are bad!")
      if (notgood==0)
        wx=opt.wx
      endif
    endif
    if (comp(opt,"off")>0)
      notgood=(dim(dim(opt.off))!=1)*(rows(opt.off)!=n)+(rows(dim(opt.off))!=1)
      error(notgood>0, "offsets in off are bad!")
      if (notgood==0)
        off=opt.off
      endif
    endif
;
    if (comp(opt,"alter")>0)
      opt.alter=ceil(opt.alter)
      notgood=((rows(opt.alter)>p)||(rows(dim(opt.alter))!=1))
      notgood=notgood|(max(opt.alter)>p)||(min(opt.alter)<1)
      error(notgood>0, "index 'alter' is bad!")
      if (notgood==0)
        alter=opt.alter
      endif
    endif
    if (comp(opt,"indiv")>0)
      opt.indiv=ceil(opt.indiv)
      notgood=((rows(opt.indiv)>p)||(rows(dim(opt.indiv))!=1))
      notgood=notgood|(max(opt.indiv)>p)||(min(opt.indiv)<1)
      error(notgood>0, "index 'indiv' is bad!")
      if (notgood==0)
        indiv=opt.indiv
      endif
    endif
;
    if ((exist(alter)==1)&&(exist(indiv)==1))
      {full,tmp}=discrete(indiv|alter)
      error(rows(full)<rows(indiv|alter),"indices in 'alter' and 'indiv' overlap!")
    endif
;
    if (comp(opt,"shf")>0)
      shf=(opt.shf==1)*(dim(dim(opt.shf))==1)
    endif
    if (comp(opt,"miter")>0)
      tmp=floor(opt.miter[1,1])
      notgood=(tmp<1)+(dim(dim(opt.miter))!=1)
      warning(notgood>0,"opt.miter was unusable, used default =10 instead")
      miter=notgood*miter+(1-notgood)*tmp
    endif
    if (comp(opt,"cnv")>0)
      tmp=opt.cnv[1,1]
      notgood=(tmp<=0)+(dim(dim(tmp))!=1)
      warning(notgood>0,"opt.cnv was unusable, used default =0.0001 instead")
      cnv=notgood*cnv+(1-notgood)*tmp
    endif
  endif
;
  if (cols(y)==1)
    {jcat,tmp}=discrete(y+0)
    y=(y==reshape (jcat, cols(jcat)|rows(jcat))) ; trans(jcat)
    m=rows(jcat)
  else
    m=cols(y)
  endif
;
  if ((exist(alter)!=1)&&(exist(indiv)!=1))
    indiv=1:p
  endif
  if (exist(alter)==1)
    q=rows(alter)/m
    error(ceil(q)!=q, "index 'alter' is bad!")
  endif
;
  p=0
  q=0
;
  switch
    case (exist(alter)&&(1-exist(indiv)))
      alter=x[,alter]
      indiv=0.*matrix(n)
      q=cols(alter)/m
      p=0
      break
    case ((1-exist(alter))&&exist(indiv))
      alter=0.*matrix(n)
      indiv=x[,indiv]
      q=0
      p=cols(indiv)
      break
    default
      alter=x[,alter]
      indiv=x[,indiv]
      q=cols(alter)/m
      p=cols(indiv)
      break
  endsw
  x=list(indiv,alter)
;
  if (p>0)
    error (countNaN(inv(x.indiv'*(wx.*x.indiv)))>0, "design is singular!")  
  endif
;
  alter=0
  indiv=0.*matrix(1,m)
  if (p>0)
    indiv=0.*matrix(p,m)
  endif
  if (q>0)
    alter=0.*matrix(q)
  endif
  b=list(indiv,alter)
;
  if (q>0)
    alts=((1:(q*m))-1)%m+1        ; alternatives (in each variable)
;    x.alter=x.alter.*((alts!=1)') ; each first x.alter variable =0
;
    idm=diag(matrix(m))
    idq=diag(matrix(q))
    mulq=kron(matrix(1,q),idm)   ; multiples from  r x m  to  r x (q*m)
;    sumq=kron(idq,0|matrix(m-1)) ; sums up   from  r x (q*m)  to  r x q
    sumq=kron(idq,1|matrix(m-1)) ; sums up   from  r x (q*m)  to  r x q
  endif
;
  switch
    case ((p>0)&&(q==0))
      tmp=exp(x.indiv*b.indiv)
      break
    case ((p==0)&&(q>0))
      tmp=exp(x.alter*kron(b.alter,idm))
      break
    default
      tmp=exp(x.alter*kron(b.alter,idm)+x.indiv*b.indiv)
      break
  endsw
  mu =tmp./sum(tmp,2)
;
  ctrl = 1 
  deviance=NaN
  loglik=NaN
;
  do 
    if (shf==1)
      "'glmmultlo': iteration no. "+string("%1.0f", ctrl)
    endif
;
    tmp=y-mu
    grad=0.*matrix(q+p*(m-1))
    if (q>0)
      grad[1:q]=(sum(x.alter.*(tmp*mulq))*sumq)'
    endif
    if (p>0)
      grad[q+1:q+p*(m-1)]=vec(x.indiv'*(tmp[,2:m]))
    endif
;grad
;
    bv=0.*matrix(q+p*(m-1),q+p*(m-1))
;
    if (q>0)
      tmp=(mu*mulq).*x.alter
      tmpx=tmp*sumq
      bv[1:q,1:q]=-tmpx'*tmpx
      k=0
      while (k<m)
        k=k+1
        ik=m*(0:(q-1))+k
        bv[1:q,1:q]=bv[1:q,1:q]+x.alter[,ik]'*tmp[,ik]
      endo
    endif
;   
    if (p>0)
      k=0  
      while (k<m-1)
        k=k+1
        k0=q+(k-1)*p+1
        k1=q+k*p
        if (q>0)
          tmpx=((mu*mulq).*x.alter)*sumq
          ik=m*(0:(q-1))+k+1
          tmp=(x.alter[,ik]-tmpx)'*(mu[,k+1].*x.indiv)
          bv[1:q,k0:k1]=tmp
          bv[k0:k1,1:q]=tmp
        endif
        l=k-1
        while (l<m-1)
          l=l+1
          l0=q+(l-1)*p+1
          l1=q+l*p
          tmp= -mu[,k+1].*mu[,l+1] + (k==l).*mu[,l+1]
          tmp=x.indiv'*(tmp.*x.indiv)
          bv[k0:k1,l0:l1]=tmp
          bv[l0:l1,k0:k1]=tmp
        endo
      endo
    endif
    bv=inv(bv)
;bv
;
    tmp=0.*matrix(q+p*(m-1))
    if (q>0)
      tmp[1:q]=b.alter
    endif
    if (p>0)
      tmp[q+1:q+p*(m-1)]=vec(b.indiv[,2:m])
    endif
    tmp=tmp+bv*grad
;
    ret=0
    if (countNaN(tmp)==0)
      if (q>0)
        b.alter=tmp[1:q]
      endif
      if (p>0)
        b.indiv=0.*matrix(p)~reshape(tmp[q+1:q+p*(m-1)],p|(m-1))
      endif
      switch
        case ((p>0)&&(q==0))
          tmp=exp(x.indiv*b.indiv)
          break
        case ((p==0)&&(q>0))
          tmp=idm
;          tmp[1,1]=0
          tmp=exp(x.alter*kron(b.alter,tmp))
          break
        default
          tmp=idm
;          tmp[1,1]=0
          tmp=exp(x.alter*kron(b.alter,tmp)+x.indiv*b.indiv)
          break
      endsw
      mu =tmp./sum(tmp,2)
      if (ctrl == 1) 
        crit = 0 
        ctrl = 2  
        deviance=2.*sum( sum(xlnxdy(y,mu),2) )
        bold = b 
        bvold = bv
        chgb=Inf
        chgd=Inf
      else 
        chgd = deviance 
        deviance=2.*sum( sum(xlnxdy(y,mu),2) )
        chgd = abs(chgd-deviance)/chgd 
        db   = vec(b.alter-bold.alter,b.indiv-bold.indiv)
        tmp  = vec(bold.alter,bold.indiv)
        chgb = sqrt((db'*db)./(tmp'*tmp)) 
        bold = b 
        bvold = bv
        crit = (((chgd<cnv)+(chgb<cnv))>0)
        ctrl = ctrl +1 
      endif
    else
      crit=1
     "'glmmultlo': missing values have been encountered"
      if (exist(bold))
        b=bold
        bv=bvold
      endif
      ret=-1
    endif 
;
    it=ctrl-1
    if (shf==1)
      switch
        case (q==0)
          b.indiv
          break
        case (p==0)
          b.alter
          break
        default
          b
          break
      endsw
      "relative change in b:        "+string("%10.8f", chgb)
      "relative change in deviance: "+string("%10.8f", chgd)
    endif
    if (1-crit)
      ret=1
    endif
  until (crit+(ctrl>miter)) 
;
  switch
    case (q==0)
      delete(b,2)
      indiv=0.*matrix(p*m,p*m)
      indiv[p+1:p*m,p+1:p*m]=bv[q+1:q+p*(m-1),q+1:q+p*(m-1)]
      bv=list(indiv)
      indiv=reshape(sqrt(xdiag(bv)),p|m)
      serror=list(indiv)
      indiv=(matrix(p)~b.indiv[,2:m])./serror.indiv
      tvalue=list(indiv)
      indiv=2.*cdfn(-abs((99.*matrix(p))~tvalue.indiv[,2:m]))
      pvalue=list(indiv)
      break
    case (p==0)
      delete(b,1)
      alter=bv[1:q,1:q]
      bv=list(alter)
      alter=sqrt(xdiag(bv))
      serror=list(alter)
      alter=b.alter./serror.alter
      tvalue=list(alter)
      alter=2.*cdfn(-abs(tvalue.alter))
      pvalue=list(alter)
      break
    default
      alter=bv[1:q,1:q]
      indiv=0.*matrix(p*m,p*m)
      indiv[p+1:p*m,p+1:p*m]=bv[q+1:q+p*(m-1),q+1:q+p*(m-1)]
      mixed=0.*matrix(p*m,q)
      mixed[p+1:p*m,]=bv[q+1:q+p*(m-1),1:q]
      bv=list(indiv,alter,mixed)
      indiv=reshape(sqrt(xdiag(bv.indiv)),p|m)
      alter=sqrt(xdiag(bv.alter))
      serror=list(indiv,alter)
      indiv=(matrix(p)~b.indiv[,2:m])./serror.indiv
      alter=b.alter./serror.alter
      tvalue=list(indiv,alter)
      indiv=2.*cdfn(-abs((99.*matrix(p))~tvalue.indiv[,2:m]))
      alter=2.*cdfn(-abs(tvalue.alter))
      pvalue=list(indiv,alter)
      break
  endsw
;
  loglik  =   sum( sum(xlny(y,mu),2) )
  n0 = sum(y)
  p0 = n0./n
  loglik0 = n0*log(p0')
  r2    = 1-loglik/loglik0
  adr2    = 1-(loglik/(n-p*(m-1)-q))/(loglik0/(n-(m-1)))
;
  stat  = list(serror,tvalue,pvalue,deviance,loglik,it,ret)
  if (r2 >0) ; no r2 without constant!
    append(stat,r2)
  endif
  if (adr2 >0)
    append(stat,adr2)
  endif
endp
   


