proc(sc)=glmscatter(code,x,y,opt)
; ----------------------------------------------------------------------------
; Library       glm
; ----------------------------------------------------------------------------
;  See_also     glminvlink
; ----------------------------------------------------------------------------
;   Macro       glmscatter
; ----------------------------------------------------------------------------
;   Description  glmscatter computes a scatterplot to explain the
;                marginal influence of a variable on the response.
; ----------------------------------------------------------------------------
;   Link  ../tutorials/glmstart.html Tutorial: GLM in XploRe 
; ----------------------------------------------------------------------------
;   Usage        sc = glmscatter(code,x,y)
;   Input
;     Parameter   code 
;     Definition        text string, the short code for the model (e.g. 
;                       "bilo" for logit or "noid" for ordinary LS).
;     Parameter   x  
;     Definition        n x 1  matrix, explanatory variable.
;     Parameter   y
;     Definition        n x 1  response.
;     Parameter   opt
;     Definition        optional, a list with optional input. The macro
;                       "glmopt" can be used to set up this parameter.
;                       The order of the list elements is not important.
;                       Parameters which are not given are replaced by 
;                       defaults (see below).
;     Parameter   opt.weights
;     Definition        string, type of weights. Can be "frequency"
;                       for replication counts, or "prior" (default)
;                       for prior weights in weighted regression.
;     Parameter   opt.wx
;     Definition        scalar or n x 1 vector, frequency or prior 
;                       weights. If not given, set to 1.
;     Parameter   opt.pow
;     Definition        optional, power for power link.
;     Parameter   opt.nbk
;     Definition        scalar, extra parameter k for negative binomial
;                       distribution. If not given, set to 1 (geometric 
;                       distribution).
;   Output
;     Parameter   sc
;     Definition        m x 2  discretized or binned x in the first
;                       column and the invers link function applied 
;                       to the averaged y (in bin) in the second column.
; ----------------------------------------------------------------------------
;   Example   library("glm")
;             x=normal(50)
;             y=(x+normal(50) >0)         ; probit model
;             sc=glmscatter("bipro",x,y)
;             library("plot")
;             m=setmask(x~x,"line","red") ; true relation = linear
;             plot(sc,m)
; ----------------------------------------------------------------------------
;   Result    The density of x, the marginal influence of x and a
;             scatter plot of x and y.
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/18
; ----------------------------------------------------------------------------
;
  glmmodels=getglobal("glmmodels")
;
  binomial  = sum(code==(glmmodels.binomial)) >0
  poisson   = sum(code==(glmmodels.poisson)) >0
  gamma     = sum(code==(glmmodels.gamma))>0
  igaussian = sum(code==(glmmodels.igaussian))>0
  nbinomial = sum(code==(glmmodels.nbinomial))>0
  power     = sum(code==(glmmodels.power)) >0
;
; now check which optional values have been given
;
  wx=1
  pow=0
  n=rows(x)
  weights="prior"
;
  if (exist(opt)>0)
    if (comp(opt,"wx")>0)
      notgood=(dim(dim(opt.wx))!=1)*(rows(opt.wx)!=n)+(rows(dim(opt.wx))!=1)
      notgood=notgood+(sum(abs(wx))==0)+(min(wx)<0)
      error(notgood>0, "weights in opt.wx are bad!")
      if (notgood==0)
        wx=opt.wx
      endif
    endif
    if (comp(opt,"pow")>0)
      notgood=(dim(dim(opt.pow))!=1)
      warning (notgood>0, "opt.pow was unusable, used default =0 instead")
      if (notgood==0)
        pow=opt.pow
      endif
    endif
    if (comp(opt,"weights")>0)
      notgood=(exist(opt.weights)!=2)
      error(notgood>0, "wrong type of weights in opt.weights!")
      if (notgood==0)
        weights=opt.weights
      endif
    endif
  endif
  wx=wx.*matrix(rows(x))
;
  if (binomial)
    if (weights=="prior")
      if (max(y[,1])>1)
        y = y[,1]./(wx+(wx==0))
      endif
    endif
  endif
;
  if (weights=="frequency")
    {xr,r}=discrete(x,wx~y)
    if (rows(r)>min(50|(n/2)))
      d=(max(x)-min(x))/(min(50|(n/10)))
      o=min(x)-d/2
      {xr,r}=bindata(x,d,o,wx~y)
      xr=paf(xr,r[,1]>0)
       r=paf( r,r[,1]>0)
      xr=o+d.*xr
    endif
    yr=r[,3]
    r =r[,2]
  else
    {xr,r}=discrete(x,y)
    if (rows(r)>min(50|(n/2)))
      d=(max(x)-min(x))/(min(50|(n/10)))
      o=min(x)-d/2
      {xr,r}=bindata(x,d,o,y)
      xr=paf(xr,r[,1]>0)
       r=paf( r,r[,1]>0)
      xr=o+d.*xr
    endif
    yr=r[,2]
    r =r[,1]
  endif   
;
  other=((power)&&(pow!=1)||gamma||poisson||igaussian||nbinomial)
;
  if (binomial||other) ; handle yr/r is 0 or 1 in binomial
                       ; handle yr/r is 0 in other
    if (binomial)
      i=(yr==0)||(r==yr)
    else
      i=(yr==0)
    endif
    switch
      case ((min(i)==0)&&(max(i)==1))
        j=paf(1:rows(i),i==0)
        j=j[rows(j)]
        i[j]=1
        if (j<rows(i))
          i[j+1:rows(i)]=0.*i[j+1:rows(i)]
        endif
        tmp=paf(cumsum(xr~yr~r~i),i-1)
        n=rows(tmp)
        if (n>1)
          xr=tmp[1,1]|(tmp[2:n,1]-tmp[1:(n-1),1])
          yr=tmp[1,2]|(tmp[2:n,2]-tmp[1:(n-1),2])
           r=tmp[1,3]|(tmp[2:n,3]-tmp[1:(n-1),3])
           i=tmp[1,4]|(tmp[2:n,4]-tmp[1:(n-1),4])
          sc=(xr/(i+1))~glminvlink(code,(yr/r))
        else
          xr=tmp[1,1]
          sc=mean(xr)~glminvlink(code,0.5)
        endif
        break
      case ((min(i)==0)&&(max(i)==0))
        sc=xr~glminvlink(code,(yr/r))
        break
      case ((min(i)==1)&&(max(i)==1))
        sc=mean(xr)~glminvlink(code,0.5)
        r=15
        break
      default ; nothing
    endsw
  else
    sc=xr~glminvlink(code,(yr/r))
  endif
  bulletsize=ceil(sqrt(r/max(r))*13)+2
  setmaskp(sc,0,8,bulletsize)    
endp




