proc(res, out)=linregres(x, y, yh)
; -----------------------------------------------------------------------
; Library      stats
; -----------------------------------------------------------------
; See_also     gls linreg linregfs
; -----------------------------------------------------------------------
; Macro        linregres
; -----------------------------------------------------------------------
; Description  linregres computes some residual analysis for a linear
;              regression.
; ----------------------------------------------------------------------
;   Notes      The return values are res, which contains the residuals
;              in the first column, the leverage in the second column,
;              the standardized residuals in the third column and the
;              Cook distance in the last column. out consists of vectors
;              of zeros and ones indicating some influential points.
;              The first column indicates all observations with a
;              leverage larger than 2*p/n and the second column all
;              points with a Cook distance larger than 1.
; -----------------------------------------------------------------------
; Usage        {res,out} = linregres (x, y, yh)
; Input
;   Parameter  x 
;   Definition n x p   regressors 
;   Parameter  y 
;   Definition n x 1   
;   Parameter  yh
;   Definition n x 1   
; Output
;   Parameter  xfs
;   Definition n x 4
;   Parameter  out
;   Definition n x 2
; -----------------------------------------------------------------------
; Example      ; loads the library stats
;              library("stats")   
;              ; reset random generator 
;              randomize(0)
;              ; generate x
;              x = normal(100, 3)
;              ; generate y
;              y = 10*x[,3]+x[,1].*x[,2]
;              ; do the forward selection
;              {xfs,bfs}=linregfs(x, y, 0.05)
;              ; compute residual number
;              {res,out}=linregres(xfs, y, xfs*bfs)
;              ; create a display fro plotting
;              disp = createdisplay(2,2)
;              ; show residual plots
;              ;    residuals               leverage   
;              ;    standardized residuals  Cook distance
;              show (disp, 1, 1, y~res[,1])
;              show (disp, 2, 1, y~res[,3])
;              show (disp, 1, 2, y~res[,2])
;              show (disp, 2, 2, y~res[,4])
; -----------------------------------------------------------------------
; Result       shows the residual plots. From the standardized residuals
;              we find three points in the bottom with an absolute value 
;              larger than 3. The leverage plot shows on the right and
;              the left sets of points with a leverage larger than 0.04
;              (thus they are influential). The Cook distance plot shows 
;              no datapoints larger than 1 and we can not find outliers
;              in x. 
; -----------------------------------------------------------------------
; Author       Sigbert Klinke   970522
; ----------------------------------------------------------------------
; Reference    Chatterjee, S. and Price, B. (1991),
;              Regression Analysis by Example, Whiley, p. 83-91
; ----------------------------------------------------------------------
  res = y-yh
  pii = xdiag(x*inv(x'*x)*x')
  esi = res./sqrt(var(res).*(1-pii))
  ci2 = pii.*esi^2/(cols(x).*(1-pii))
  res = res~pii~esi~ci2 
  out = (pii>2.*cols(x)/rows(x))~(ci2>1)
endp