proc (mh) = lpregxest(x,h,p,v) 
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      lpregest sker locpol
; -----------------------------------------------------------------
;   Macro        lpregxest
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel regression, 
;                local polynomial estimator
; -----------------------------------------------------------------
;   Description  estimates a univariate regression function using 
;                local polynomial kernel regression with
;                Quartic kernel.
; -----------------------------------------------------------------
;   Notes        This function does an exact computation, i.e.
;                requires O(n^2) operations for estimating the 
;                regression function on all observations. For
;                exploratory purposes, the macro "regest" is
;                recommended, which uses the faster WARPing method.
; -----------------------------------------------------------------
;   Reference    Fan and Gijbels (1995): Local Polynomial Fitting
;
;                Fan and Marron (1994): Binning for local polynomials
; 
;                Haerdle (1991): Smoothing Techniques
; -----------------------------------------------------------------
;   Usage        y = lpregxest (x,h {,p {,v}})
;   Input
;     Parameter  x  
;     Definition   n x 2, the data. In the first column the
;                  independent, in the second column the
;                  dependent variable.
;     Parameter  h 
;     Definition   scalar, bandwidth. If not given, the rule of thumb
;                  bandwidth computed by lpregrot is used.
;     Parameter  p
;     Definition   integer, order of polynomial. If not given,
;                  p=1 (local linear) is used. p=0 yields the 
;                  Nadaraya-Watson estimator. p=2 (local
;                  quadratic) is the highest possible order.
;     Parameter  v  
;     Definition   m x 1, values of the independent variable on 
;                  which to compute the regression. If not given, 
;                  x is used.
;   Output
;     Parameter  mh  
;     Definition   n x 2 or m x 2 matrix, the first column is the 
;                  sorted first column of x or the sorted v, the 
;                  second column contains the regression estimate  
;                  on the values of the first column.
; -----------------------------------------------------------------
;   Example   library("smoother") 
;             library("plot")
;             ;
;             x = 4.*pi.*(uniform(200)-0.5)   ; independent variable
;             m = cos(x)                      ; true function
;             e = uniform(200)-0.5            ; error term             
;             x = x~(m+e)                             
;             ;
;             mh = lpregxest(x,1)                ; estimate function
;             ;
;             mh = setmask(mh, "line","blue")
;             m  = setmask(sort(x[,1]~m) , "line","black","thin")
;             plot(x,mh,m)                                         
; -----------------------------------------------------------------
;   Result    The Nadaraya-Watson regession estimate (blue) using   
;             Quartic kernel and bandwidth h=1 and the true 
;             regression function (thin black) are pictured.
; -----------------------------------------------------------------
;   Author    Marlene Mueller, 980413
; -----------------------------------------------------------------
  n=rows(x)
  error(cols(x)<>2,"lpregxest: cols(x) <> 2")
;
  if (exist(h)==0)
    h=lpregrot(x)
  endif
  if (exist(p)==0)
    p=1            ; default is local linear    
  endif    
  error(p>2,"lpregxest: p must be <= 2")
;    
  tmp=sort(x)
  x=tmp[,1]
  y=tmp[,2]
  if (exist(v)==0)
    v=x
  else
    v=sort(v)
  endif
;
  if (p==0)
    mh=sker(x,h,"qua",matrix(n)~y,v)
    mh=v~ (mh[,2]./mh[,1])
  else
    mh=locpol(x, v, y, h, p-1) 
    mh=v~mh[,1]
  endif
endp  


