proc(b) = lts(x, y, h, all, mult)
; ----------------------------------------------------------------------------
; Library       metrics
; ----------------------------------------------------------------------------
;   Macro       lts
; ----------------------------------------------------------------------------
;   Description   Computes the least trimmed squares estimate for the 
;		  coefficients of a linear model. 
; ----------------------------------------------------------------------------
;   Keywords    Robust regression
; ----------------------------------------------------------------------------
;   Usage       b = lts(x, y{, h, all, mult})
;
;   Input
;     Parameter   x  
;     Definition        n x p design matrix of explanatory variables.
;     Parameter   y
;     Definition        n x 1 vector, dependent variable.
;     Parameter   h  
;     Definition        optional, trimming constant; default value is 
;                       [n/2]+[(p+1)/2], where n is the number of observations
;                       and p is the number of parameters. h should belong to 
;                       {[(n+1)/2],...,n} and must be bigger than p.
;     Parameter   all
;     Definition        optional, logical flag for the exact computation (nonzero = TRUE),
;                       default value is 0 (FALSE). If ci = 0, an approximation
;                       of the estimator is computed. If ci != 0, the estimate
;                       is computed precisely - this can be rather time demanding
;                       and applicable only for small sample sizes. The number of
;                       iterations corresponds in this case to n over h.
;     Parameter   mult
;     Definition        optional, affects the maximal number of iterations, 
;                       after which the algorithm is stoped (if convergence
;                       was not achieved earlier), default value equals to 1. 
;                       The maximal number of iterations is currently 
;                       '(600 * h) * times', so this variable allow
;                       to reduce or extend this number.
;
;   Output
;     Parameter   b
;     Definition        p x 1 vector of estimated coefficients.
; ----------------------------------------------------------------------------
;   Example   library("metrics")
;             ; 
;             ; simulate data
;             ;
;             randomize(555)
;             x = uniform(50,3)
;             y = x[,1] + 2*x[,2] - x[,3] + normal(50)
;             ;
;             ; fit the data...
;             ;
;             b = lts(x,y)
;             b
; ----------------------------------------------------------------------------
;   Result    Contents of coefs - estimates of b = (1,2,-1)' coefficient vector
;             [1,]  0.95542 
;             [2,]  2.0811 
;             [3,] -0.25978   
; ----------------------------------------------------------------------------
;   Example   ; procedure for graphical representation of the results estimated by rqfit quantlet
;             ; parameter obs1 is an outlier added to randomly generated data points
;             proc() = myquant(obs1)
;                 ;
;                 ; initialization
;                 ;
;                 n = 10                   ; number of observations
;                 randomize(17654321)      ; sets random seed
;                 beta = #(1, 2)           ; defines intercept and slope
;                 x = matrix(n)~sort(uniform(n))  
;                 ;                        ; creates design matrix   
;                 ;
;                 ; new x-observation is added
;                 ;
;                 x = x|(1~obs1[1])
;                 m = x*beta               ; defines regression line
;                 eps = 0.05*normal(n)     ; creates obs error
;                 ;
;                 ; new y-observation is added
;                 ;
;                 y = m[1:n] + eps         ; noisy line
;                 y = y|obs1[2]
;                 ;
;                 ; create graphical display and draw data points
;                 ;
;                 d = createdisplay(1,1)     
;                 dat = x[,2]~y                              
;                 outl = obs1[1]~obs1[2]
;                 setmaskp(outl,1,12,15)    ; outlier is blue big star
;                 tdat = x[,2]~m
;                 setmaskl(tdat, (1:rows(tdat))', 1, 1, 1) 
;                 ;                        ; thin blue line
;                 setmaskp(tdat, 0, 0, 0)  ; reduces point size to min
;                 ;
;                 ; estimation of the model using rqfit 
;                 ;
;                 beta1 = lts(x,y)
;                 ;
;                 ; draw estimated regression line
;                 ;
;                 yhat = x*beta1
;                 hdat = x[,2]~yhat
;                 setmaskp(hdat, 0, 0, 0)    
;                 setmaskl(hdat, (1:rows(hdat))', 4, 1, 3)
;                 ;                        ; thick red line
;                 show(d, 1, 1, dat[1:n], outl, tdat, hdat)
;                 title="LTS regression with an outlier"
;                 setgopt(d,1,1,"title",title) 
;                 ;                        ; sets title 
;             endp                         ; end of myquant
;             ;
;             ; load metrics library
;             ;
;             library("metrics")
;             ;
;             ; call estimation function with outlier #(0.9,4.5)
;             ;
;             myquant(#(0.9,4.5))
; ----------------------------------------------------------------------------
;   Result    As a result, you should see a graph, in which observations are
;             denoted by black circles and an outlier (observation #(0.9,4.5))
;             is represented by the blue big star in the right upper corner of
;             the graph. 
;             The blue line depicts the true regression line (beta = #(1,2)),
;             while the thick red line shows the estimated regression line.
; ----------------------------------------------------------------------------
;   References
;   	      Rousseeuw, P.J. and Leroy, A.M. (1987):
;             Robust Regression and Outlier Detection, J. Wiley & Sons.
;
;             Visek, J. A. (2000): On the diversity of estimates.
;             To appear in {\it Computational Statistics and Data Analysis}.
; ----------------------------------------------------------------------------
;   Author   Pavel Cizek, 2000-02-01
; ----------------------------------------------------------------------------
;
; load dll
  if (getenv("os") == "unix")
    handle = dlopen("lts.so")
  else
    handle = dlopen("lts.dll")
  endif
;
; check parameter existence and their values
  error(exist(x) <> 1, "Missing data matrix!")
  error(exist(y) <> 1, "Missing dependent variable!")
  error(rows(x) <> rows(y), "x and y have different dimensions!")
  p = cols(x)
  n = rows(x)
  error(n <= p, "Number of observations less than number of parameters!")
  
  if (exist(h) <> 1)
    h = floor(n/2) + floor((p + 1)/2)
  endif
  error((h < (n+1)/2) || (n < h) || (h <= p), "Wrong choice of trimming constant h!")
  if (exist(all) <> 1)
    all = 0
  endif
  if (exist(mult) <> 1)
    mult = 1
  endif
  if (mult <= 0)
    mult = 1
  endif
; prepare parameters for call
; number of iterations - ask if it is too high!!! to be done!!!
  if (all <> 0)  ; compute n over h
    cit = n;
    jmen = h;
    numiters = cit / jmen;
    while (jmen > 1)
      cit = cit - 1
      jmen = jmen - 1
      numiters = numiters * cit / jmen
    endo
  else  
    numiters = h * 600 * mult;
    ; not needed ... 500 replaced by 600   numiters = numiters * 1.2;	// originally 1.1
  endif
; check whether the number of iterations is not too high
  if (numiters > 500000)
     headline = "Number of iterations is too high!"
     items = "Continue (it takes some time)" | "Cancel computation"
     i = selectitem(headline, items)
     error(i[2] <> 0, "The computation was canceled by user!")
  endif
; allocation of the memory
; EXTERN long EXPORT LTS(double *b, double *yx,
;                        double *n, double *p, double *h,
;                        double *allposibilities, double *numiters,
;                        double *tenarrays, double *matrices, double *vectors)
; b ... beta vector p x 1
  b = 0 * matrix(p)
; yx ... data matrix n x p+1
  yx = y~x
; n, p, h, all, numiters are already initialized
; tenarrays ... 22 * (p + 2)
  tenarrays = 0 * matrix(22*(p+2))
; matrices ... 4*p^2
  matrices = 0 * matrix(p, 4*p)
; vectors ... 5*n + 2*p
  vectors = 0 * matrix(5*n + 2*p)
; call dll
  if (getenv("os") == "unix")
    ret = dlcall(handle, "LTS", b,yx,n,p,h,all,numiters,tenarrays,matrices,vectors)
  else
    ret = dlcall(handle, "_LTS", b,yx,n,p,h,all,numiters,tenarrays,matrices,vectors)
  endif
  
; get results - stored in b; it ret == 0, no model was estimated...
  warning(ret == 0, "It is impossible to estimate this model!")
  if (ret == 0)
    beta = 0 * matrix(p)
  endif
; close dll
  dlclose(handle)
;end
endp

