proc(r,s) = numint2(aIh, b, N)

; -----------------------------------------------------------------
; Library        kalman
; -----------------------------------------------------------------
;  See_also      rICfil, calibrIC,  ICerz
; -----------------------------------------------------------------
;   Macro        numint2
; -----------------------------------------------------------------
;   Description  Auxiliary routine for rICfil:
;                calculates for dimension p=2 
;                diag(E[ YY'  u min(b/|aIhY|,u) ]) 
;   and          diag(E[ YY'   min(b/|aIhY|,u)^2 ]) 
;                for u square root of a Chi^2_2-variable, 
;                and Y~ufo(S_2) indep of u
;                by using a polar representation of
;                Lambda:= I^{1/2} Y u,  u = | I^{-1/2} Lambda |,  
;                Y=I^{-1/2} Lambda /u
;
;                the integrals are evaluated stepwise, first conditioning on Y
;                and calculated "analytically" using Ewinn, Ew2inn
;                and then the outer integration is done by a Romberg-Procedure 
;                along the directions Y, parametrized by a 
;                sin-cos-representation.
; -----------------------------------------------------------------
;   Keywords     robust Kalman filter, rIC
; -----------------------------------------------------------------
;   Usage          {r,s}=numint2(aIh,b,N)
;   Input
;     Parameter   aIh
;     Definition  dim: 2; diag. of Eigenvalues of A I^{1/2}, 
;                 I Fisher-Information (>=0)
;     Parameter   b 
;     Definition     numeric; clipping height (>0)
;     Parameter   N
;     Definition     numeric; number of gridpoints for Integration
; Output                                                           
;     Parameter   r
;     Definition     2-dim ; r=diag(E[ YY'  u min(b/|aIhY|,u) ])
;     Parameter   s
;     Definition     2-dim; s=diag(E[ YY'  min(b/|aIhY|,u)^2 ])
; -----------------------------------------------------------------
;  Notes 
;    Not to be used outside of rICfil!
; 
;      Is called by itera.
; 
;      Calles [within kalman.lib]  ew2inn ewinn.
; -----------------------------------------------------------------
;   Example   to be looked up in rICfil
; -----------------------------------------------------------------
;   Author    P.Ruckdeschel 991010 
; -----------------------------------------------------------------

   
; Gridpoints
Ln=ceil(log(N)/log(2.0))
n=2^(Ln)

th=(0:n)'/n*2*pi


; Romberg-Tableaux
T1=matrix(2,Ln+1)*0
T2=matrix(2,Ln+1)*0

h=th[1,n+1]-th[1,1]

Y=matrix(2,n+1)
Y[1,]=cos(th)
Y[2,]=sin(th)
Y2=aIh.*Y
nY2=sqrt(Y2[1,]^2+Y2[2,]^2)
nY2=(nY2>0).*nY2+(nY2==0).*0.03
c=b/nY2

ri=ewinn(c,2)
si=ew2inn(c,2)

r0=Y^2.*ri/pi/2
s0=Y^2.*si/pi/2

T1[,1]=0.5*h*(r0[,1]+r0[,n+1])
T2[,1]=0.5*h*(s0[,1]+s0[,n+1])

m=1
i=1
S=n
do
   sumr=0
   sums=0
   h=h*0.5
   S=floor((S+.5)/2)
   m=2*m   
   q=1
   j=1
   while (j<m)
       sums=sums+s0[,j*S]
       sumr=sumr+r0[,j*S]
       j=j+2
   endo
   T1[,i+1]=0.5*T1[,i]+sumr*h
   T2[,i+1]=0.5*T2[,i]+sums*h
   j=i
   while (j>1)
      q=q*4
      T1[,j-1]=T1[,j]+(T1[,j]-T1[,j-1])/(q-1)
      T2[,j-1]=T2[,j]+(T2[,j]-T2[,j-1])/(q-1)
      j=j-1
   endo
   i=i+1
until (S == 1)
r=T1[,1]
s=T2[,1]
endp
