proc() = plothist(inX, inTicks)
; ---------------------------------------------------------------------
; Library     plot
; ---------------------------------------------------------------------
; See_also    plotbox plotdot
; ---------------------------------------------------------------------
; Macro       plothist
; ---------------------------------------------------------------------
; Description Plots a histogram of x
; ---------------------------------------------------------------------
; Usage       plothist(inX {,inTicks})
; Input       
; Parameter   inX
; Definition  n x 1 Vector of continuous data.
; Parameter   inTicks
; Definition  m x 1 Vector. Specifying the (m-1) intervalls for the 
;             histogram.
; Output      
; ---------------------------------------------------------------------
; Notes       The intervalls are taken to be (inTicks[1], inTicks[2]),
;             (inTicks[2], inTicks[3]), ..., (inTicks[m-1], inTicks[m]),
;             where each left margin is element of the intervall and each
;             right margin is not.
;
;             inTicks[i] < inTicks[i+1] must hold for all i=1, ..., m-1.
; ---------------------------------------------------------------------
; Example     library("plot")
;             x = normal(200)
;             ticks = 2*(-3:3)
;             plothist(x, ticks)
; ---------------------------------------------------------------------
; Result      Shows a histogram of x.
; ---------------------------------------------------------------------
; Keywords    histogram, high level graphics 
; ---------------------------------------------------------------------
; Author      Stephan R. W. Lauer, 990303
; ---------------------------------------------------------------------

   if (exist("inX") <> 1)
      error(1, "plothist: The data must be numerical")
   endif

   if (exist("inTicks") <> 1)
      // no or no valid tickmarks specified, calculate our own ones
      // use by default 12 equidistant groups, anchored at integers
      inTicks = 0:12 // init inTicks
      inTicks = inTicks .* (ceil(max(inX))-floor(min(inX)))/12
      inTicks = inTicks + floor(min(inX))
   endif

   theNumberOfBins = rows(inTicks) - 1
   theNumberOfCases = rows(inX) 

   // now calculate the vector of heights and widths of the bins
   // the widths are given via inTicks, the heights have to be 
   // calculated from the frequencies of inX

   theWidths = matrix(theNumberOfBins)
   theHeights = matrix(theNumberOfBins)   
 
   count = 1
   while (count <= theNumberOfBins)
         theWidths[count] = inTicks[count+1] - inTicks[count]
         theFrequency = sum((inX >= inTicks[count]) && (inX < inTicks[count + 1])) / theNumberOfCases
         theHeights[count] = theFrequency / theWidths[count]  
   count = count + 1
   endo

   // now set up the boxes

   //  UpperLeft (UL) +--+ UpperRight (UR)
   //                 |  |
   //                 |  |
   //  LowerLeft (LL) +--+ LowerRight (LR)

   theLLCorners = inTicks[1:theNumberOfBins] ~ (0 * matrix(theNumberOfBins))
   theLRCorners = inTicks[2:(theNumberOfBins + 1)] ~ (0 * matrix(theNumberOfBins))
   theULCorners = inTicks[1:theNumberOfBins] ~ theHeights[1:theNumberOfBins]
   theURCorners = inTicks[2:(theNumberOfBins + 1)] ~ theHeights[1:theNumberOfBins]

   thePoints = theLLCorners | theLRCorners | theURCorners | theULCorners

   // connect LL--LR
   theLines = (1:theNumberOfBins) ~ ((theNumberOfBins + 1):(2*theNumberOfBins))
   
   // connect LR--UR
   theLines = theLines | (((theNumberOfBins + 1):(2*theNumberOfBins)) ~ ((2*theNumberOfBins + 1):(3*theNumberOfBins)))

   // connect UR--UL
   theLines = theLines | (((2*theNumberOfBins + 1):(3*theNumberOfBins)) ~ ((3*theNumberOfBins + 1):(4*theNumberOfBins)))

   // connect UL--LL
   theLines = theLines | (((3*theNumberOfBins + 1):(4*theNumberOfBins)) ~ (1:theNumberOfBins))


   // set points and lines
   setmaskp(thePoints, 0, 0, 8) // invisible
   setmaskl(thePoints, theLines, 0, 1, 1) // black, solid

   // show results
   histogram = createdisplay(1, 1)
   show(histogram, 1, 1, thePoints)
  
endp