proc() = plotlorenz(X)
; ---------------------------------------------------------------------
; Library     plot
; ---------------------------------------------------------------------
; See_also    lorenz
; ----------------------------------------------------------------------------
; Macro       plotlorenz
; ---------------------------------------------------------------------
; Keywords    Gini coefficient, Lorenz curve, concentration
; ---------------------------------------------------------------------
; Reference   Kotz/Johnson (1985): Encyclopedia of Statistical
;             Sciences, vol 5, p.156f.
; ---------------------------------------------------------------------
; Description plots a Lorenz curve
; ---------------------------------------------------------------------
; Usage       plotlorenz(X)
;   Input       
;     Parameter     X (must be positive)
;       Definition    n x 1 vector of original data or
;                     n x 2 matrix of classified data:
;                     first column = class means
;                     second column = frequencies.
; ---------------------------------------------------------------------
; Example     library("plot")
;             x = normal(200)
;             x = abs(x)
;             plotlorenz(x)
;             y = 1:5 ~ #(2,6,8,5,2)
;             plotlorenz(y)
; ---------------------------------------------------------------------
; Result      shows Lorenz curves and Gini coefficients
; ---------------------------------------------------------------------
; Author      Dandy Kleybrink, 990628; Knut Bartels, 990921
; ---------------------------------------------------------------------
  error(cols(X)>2,"input can have maxixmal 2 columns")
  Number = rows(X)
  if(cols(X)==1)
    X = X ~ matrix(Number)
  endif
  error(sum(sum(X.>=0),2)!=2*Number,"input must be nonnegative")
;
  X = sort(X,1)
  X = paf(X,(X[,2]<>0))
;
  p = X[,2]/sum(X[,2])  // relative frequencies, needed for Gini
  X[,1] = X[,1].*X[,2]
;
  X = cumsum(X)
  X = X / X[rows(X)]
;
  X = #(0,0)' | X[,2|1]
;
;----------------------------------
; calculate Gini coefficient:
;----------------------------------
  n = rows(X)
  G= 1-(X[1:(n-1),2].+X[2:n,2])'*p
  gini = 0.23~0.9  // prepare plot
  setmaskt(gini,string("Gini coefficient: %1.3f",G),1,3,20)
  setmaskp(gini,0,0,0)
;
  X = setmask(X,"line","blue")
  reference=setmask(#(0,0)'|#(1,1)',"line")
;
  plotlorenz = createdisplay(1,1)
  show(plotlorenz,1,1,reference,X,gini)
  xlabel="cumulated distribution function"
  ylabel="cumulated relative concentration"
  setgopt(plotlorenz,1,1,"title","Lorenz curve","xlabel",xlabel,"ylabel",ylabel,"xmajor",0.25,"ymajor",0.25)
endp