proc(z)=readm(filename, separators)
; -----------------------------------------------------------------------
; Library      xplore
; -----------------------------------------------------------------------
; See_also     read, write, readascii
; -----------------------------------------------------------------------
; Macro        readm
; -----------------------------------------------------------------------
; Description  readm reads mixed data from a file. 
; -----------------------------------------------------------------------
; Usage        z = readm (filename {, separators})
; Input        
;   Parameter  filename
;   Definition name of the data file
;   Parameter  separators
;   Definition string that contains the data separator (default=" \t\n")
; Output       
;   Parameter  z.type
;   Definition (p+q) vector	types of columns in file
;   Parameter  z.double
;   Definition n x p matrix	numerical vectors
;   Parameter  z.text
;   Definition n x q matrix	text vectors
; -----------------------------------------------------------------------
; Notes        If you read mixed data from a file the original position
;   of the vector in the file gets lost. Thus you will receive 3
;   matrices which will contain the vectors of one type. It is
;   ensured that, e.g. z.double[,1] was in the file before 
;   z.double[,2]. The last parameter z.type contains the recognized type t
;   of the corresponding vector in the file. A 1 represents text, a 2 an
;   integer and a 3 a float vector. The file "mixed.dat" contains
;   the following vectors:
;   &nl;
;   &nl; 1 2 3   4 hallo         4.5  .
;   &nl;5 6 7.0 0 hallo_again 6.7 8.9
;   &nl;
;   &nl;
;   The first two vector will be recognized as integer, the next vector
;   will be a float vector because of the 7.0. The fourth vector is again
;   an integer vector. The fifth vector is a text vector; if the text
;   is enclosed in apostrophs blanks can be incorporated in the text.
;   The last two vectors are float vectors. The single dot will be 
;   interpreted as NAN.
;
; -----------------------------------------------------------------------
; Example  
;          library ("xplore")    
;          readm ("mixed.dat")	   
; -----------------------------------------------------------------------
; Result   
;          Contents of string
;          [1,] "readm: Found    3 line(s) and    7 column(s)"
;          Contents of z.type
;          [1,]        0 
;          [2,]        0 
;          [3,]        0 
;          [4,]        0 
;          [5,]        1
;          [6,]        0 
;          [7,]        0  
;          Contents of z.double
;          [1,]        1        2        3        4      4.5     +NAN 
;          [2,]        5        6        7        0      6.7      8.9 
;          Contents of z.text
;          [1,] "hallo"
;          [2,] "hallo_again"      
; -----------------------------------------------------------------------
; Keywords read,file I/O
; -----------------------------------------------------------------------
; Author   Sigbert Klinke, 980515    
; -----------------------------------------------------------------------
  if (exist("separators")<>2)
    separators = " \t\n"
  endif
  {data, type} = readascii(filename, separators)
  nl = paf(1:rows(type), type==20) 
; find first elem and last elem in each line
  bl = 1|(nl+1)
  el = (nl-1)|rows(type)
; eleminate blank lines
  be = paf(bl~el, el-bl>=0)
; check for 
  dl = be[,2]-be[,1]
  dl = dl<>dl[1];
  if (sum(dl))
    dl = paf(1:rows(dl), dl)
    string ("readm: In or after line %4.0f more elements than in first line", dl)
    error(1, "readm: Different number of elements per line in file")
  endif
; construct pos matrix
  pos = paf(1:rows(type), type<>20)
  dl  = be[1,2]-be[1,1]+1
  pos=reshape (pos, #(dl, rows(pos)/dl))'
; construct type matrix
  type = paf(type, type<>20)
  dl = be[1,2]-be[1,1]+1
  type=reshape (type, #(dl, rows(type)/dl))'.>1
  string ("readm: Found %4.0f line(s) and %4.0f column(s)", rows(type), cols(type))
; check types
  dl = sum(abs(type-type[1,]), 2)
  if (sum(dl))
    dl = paf(1:rows(dl), dl)
    t = string ("readm: In or after line %4.0f element type different than in first line", dl)
    t = t|"      assuming text for this column(s)"
    t
  endif
; build result matrix
  type = max(type)'
  pn = 0
  ps = 0
  i  = 0
  while (i<rows(type))
    i = i+1
    if (type[i])
      if (ps==0)
        text = data[pos[,i]]
      else
        text = text~data[pos[,i]]
      endif
      ps = ps+1
    else
      if (pn==0)
        double = atof(data[pos[,i]])
      else
        double = double~atof(data[pos[,i]])
      endif
      pn = pn+1
   endif
  endo  
  z = list(type)
  if (pn)
    append (z, double)
  endif
  if (ps)
    append (z, text)
  endif
endp
