proc(hcrit,crit)=regbwsel(x,h,K,d)
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      regest regci regcb regbwcrit regbwrot canker
; -----------------------------------------------------------------
;   Macro        regbwsel
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel regression,
;                Nadaraya-Watson estimator, bandwidth selection
; -----------------------------------------------------------------
;   Description  interactive tool for bandwidth selection in  
;                univariate kernel regression estimation.
; -----------------------------------------------------------------
;   Reference    Haerdle (1990): Apllied Nonparametric Regression
; -----------------------------------------------------------------
;   Usage        {hcrit,crit}= regbwsel(x {,h {,K} {,d} })
;   Input
;     Parameter  x  
;     Definition     n x 2 vector, the data.  
;     Parameter  h  
;     Definition     m x 1 vector, vector of bandwidths.
;     Parameter  K  
;     Definition     string, kernel function on [-1,1] or Gaussian
;                    kernel "gau". If not given, "qua" is used.
;     Parameter  d  
;     Definition     scalar, discretization binwidth. d must be smaller
;                    than h. If not given, the minimum of min(h)/3 and 
;                    (max(x)-min(x))/200 is used.
;   Output
;     Parameter  hcrit  
;     Definition     p x 1 vector, selected bandwidths by the
;                    different criteria.
;     Parameter  crit  
;     Definition     p x 1 string vector, criteria considered for bandwidth
;                    selection.
; -----------------------------------------------------------------
;   Example   library("smoother")
;             x=read("motcyc")
;             tmp=regbwsel(x)
; -----------------------------------------------------------------
;   Result    You may interactively choose the bandwidth
;             selector. The parameters (range of bandwidths,
;             kernel K, binwidth d) can be changed as well.
;-----------------------------------------------------------------
;   Author    Marlene Mueller, 990413
; ----------------------------------------------------------------- 
  error (cols(x)<>2,"regbwsel: cols(x) <> 2")
  error (rows(x).<5,"regbwsel: at least 5 observations are required")    
  m=10      ; number of grid points  
  blanks="                              "
;
  if (exist(h)==0)
    roth=(max(x[,1])-min(x[,1]))./10
    hupp=4*roth 
    hlow=roth./4
    rangeofh=hlow|hupp
  else
    rangeofh = min(h)|max(h)
  endif
;
  if (exist(K)==0)
    K="qua"
  endif
  if (exist(K)==1) ; no K but d
    d=K
    K="qua"
  endif
;
  if (exist(d)==0)
    d=(max(x[,1])-min(x[,1]))./200
    d=min(d~(min(hlow)./3),2)
  endif
; 
  while (hupp<d)
    hupp = 2*hupp
  endo
  while (hlow<d)
    hlow = 2*hlow
  endo
  if (hupp>rangeofh[2])
    "regbwsel: upper bound for h was set to "+string("%10.6g",hupp)
  endif
  if (hlow>rangeofh[1])
    "regbwsel: lower bound for h was set to "+string("%10.6g",hlow)
  endif
  rangeofh = hlow|hupp
; 
  step=(hupp./hlow)            ; the bandwidth grid  
  step=log(step)./(m-1) 
  h=grid(log(hlow),step,m) 
  h=exp(h)
;
  stop=0
  while (stop==0)
    chra=0
    optionlist =            "Cross Validation"
    optionlist = optionlist|"Shibata's Model Selector"
    optionlist = optionlist|"Akaike's Information Criterion"
    optionlist = optionlist|"Final Prediction Error"
    optionlist = optionlist|"Rice's T"
    optionlist = optionlist|"(Change parameters)"
    optionlist = optionlist|"(Stop)"
    options = selectitem("Choose from the following", optionlist) 
    while (sum(options).>1)
      options = selectitem("Choose ONE from the following", optionlist)
    endo
    chosen=sum(options.*(1:rows(options)))
;
    switch
      case (chosen==1) ; Cross Validation
        {hopt,ch}=regbwcrit("gcv",x,h,K,d)
        cc=substr(optionlist[1]+blanks,1,28)
        if (exist("hcrit")==0)
          hcrit=hopt
          crit=cc
        else
          if (sum(crit==cc)==0)
            tmp=sum((substr(optionlist[1:5]+blanks,1,28)==crit').*(1:5),2)
            tmp=paf(tmp,tmp)
            tmp=sort((hcrit|hopt)~(tmp|1),2)
            hcrit=tmp[,1]
            crit =substr(optionlist[tmp[,2]]+blanks,1,28)
          else
            tmp=sum((crit==cc).*(1:rows(crit)))
            hcrit[tmp]=hopt
          endif
        endif
hcrit
        break
      case (chosen==2) ; Shibata's Model Selector
        {hopt,ch}=regbwcrit("shi",x,h,K,d)
        cc=substr(optionlist[2]+blanks,1,28)
        if (exist("hcrit")==0)
          hcrit=hopt
          crit=cc
        else
          if (sum(crit==cc)==0)
            tmp=sum((substr(optionlist[1:5]+blanks,1,28)==crit').*(1:5),2)
            tmp=paf(tmp,tmp)
            tmp=sort((hcrit|hopt)~(tmp|2),2)
            hcrit=tmp[,1]
            crit =substr(optionlist[tmp[,2]]+blanks,1,28)
          else
            tmp=sum((crit==cc).*(1:rows(crit)))
            hcrit[tmp]=hopt
          endif
        endif
        break
      case (chosen==3) ; Akaike's Information Criterion
        {hopt,ch}=regbwcrit("aic",x,h,K,d)
        cc=substr(optionlist[3]+blanks,1,28)
        if (exist("hcrit")==0)
          hcrit=hopt
          crit=cc
        else
          if (sum(crit==cc)==0)
            tmp=sum((substr(optionlist[1:5]+blanks,1,28)==crit').*(1:5),2)
            tmp=paf(tmp,tmp)
            tmp=sort((hcrit|hopt)~(tmp|3),2)
            hcrit=tmp[,1]
            crit =substr(optionlist[tmp[,2]]+blanks,1,28)
          else
            tmp=sum((crit==cc).*(1:rows(crit)))
            hcrit[tmp]=hopt
          endif
        endif
        break
      case (chosen==4) ; Final Prediction Error
        {hopt,ch}=regbwcrit("fpe",x,h,K,d)
        cc=substr(optionlist[4]+blanks,1,28)
        if (exist("hcrit")==0)
          hcrit=hopt
          crit=cc
        else
          if (sum(crit==cc)==0)
            tmp=sum((substr(optionlist[1:5]+blanks,1,28)==crit').*(1:5),2)
            tmp=paf(tmp,tmp)
            tmp=sort((hcrit|hopt)~(tmp|4),2)
            hcrit=tmp[,1]
            crit =substr(optionlist[tmp[,2]]+blanks,1,28)
          else
            tmp=sum((crit==cc).*(1:rows(crit)))
            hcrit[tmp]=hopt
          endif
        endif
        break
      case (chosen==5) ; Rice's T
        {hopt,ch}=regbwcrit("rice",x,h,K,d)
        cc=substr(optionlist[5]+blanks,1,28)
        if (exist("hcrit")==0)
          hcrit=hopt
          crit=cc
        else
          if (sum(crit==cc)==0)
            tmp=sum((substr(optionlist[1:5]+blanks,1,28)==crit').*(1:5),2)
            tmp=paf(tmp,tmp)
            tmp=sort((hcrit|hopt)~(tmp|5),2)
            hcrit=tmp[,1]
            crit =substr(optionlist[tmp[,2]]+blanks,1,28)
          else
            tmp=sum((crit==cc).*(1:rows(crit)))
            hcrit[tmp]=hopt
          endif
        endif
        break
      case  (chosen==6) ; Change h
        rangetitle="Lower limit for h"|"Upper limit for h"
        rangetitle=rangetitle|"Grid points for h"|"Kernel K"|"Binwidth d"
        rangevalue=string("%-1.8g",rangeofh|m)|K|string("%-1.8g",d)
        notok=1
        while (notok)
          tmp=readvalue(rangetitle,rangevalue)
          r=atof(tmp[1:3|5])
          notok=(r[1]<=0)||(r[1]>=r[2])||(exist(tmp[4])<0)
          notok=notok||(r[4]>=r[1])||(r[4]>=r[1])
          if (r[1]<=0)
            "regbwsel: lower limit for h must be positive!"
          endif
          if (r[1]>=r[2])
            "regbwsel: lower limit for h must be smaller than upper limit!"
          endif
          if (exist(tmp[4])<0)
            "regbwsel: kernel function "+tmp[4]+" does not exist"
          endif
          if (r[4]>=r[1])
            "regbwsel: d must be smaller than lower limit for h!"
          endif
          if (r[4]<=0)
            "regbwsel: d must be positive!"
          endif
        endo
        rangeofh=r[1:2]
        m=r[3]
        K=tmp[4]
        d=r[4]
        hupp=r[2]
        hlow=r[1]
        step=(hupp./hlow)            ; the bandwidth grid  
        step=log(step)./(m-1) 
        h=grid(log(hlow),step,m) 
        h=exp(h)
        chra=1
        break
      default;
        stop=1
        break
    endsw
;
    hNum=1 ; hopt might not be defined ...
    if (exist(hopt)==1)
      if (hopt==NaN)
        hNum=0
      endif
    endif
;
    if (stop==0 && chra==0 && hNum)
      ComputedCriteria =  substr(crit+blanks,1,28)+string(" ... hopt = %10.6g",hcrit)
      ComputedCriteria 
      RegressionBandwidthSelection=createdisplay(2,2)
      setfractions(RegressionBandwidthSelection, 1|2, 7|3)
;
      range=" "|" Range of h:"|string("  %-10.6g", rangeofh)
      range=range|" "|" Points:"|string("  %-10.6g", m)
      range=range|" "|(" ---------- ")
      range=range|" "|" Kernel K:"|("  "+K)
      range=range|" "|" Binwidth d:"|string("  %-10.6g", d)
      opt=" "|" optimal h:"|""|string("  %-10.6g", hopt)
      show (RegressionBandwidthSelection,1,2,opt)
;
      zeroline=rangeofh~(0|0)
      setmaskp(zeroline,0,0,0)
      setmaskl(zeroline,(1:rows(zeroline))',0,2,1)
;      
      switch
        case (chosen<=5) ; minimize criterion
          show (RegressionBandwidthSelection,2,2,range)
          setmaskp(ch,0,0,0)
          setmaskl(ch, (1:rows(ch))', 1, 1, 1)
          show (RegressionBandwidthSelection,1,1,ch) 
          setgopt(RegressionBandwidthSelection,1,1,"title",optionlist[chosen],"xlabel","h","ylabel","criterion") 
          break
        default; ROT
          show (RegressionBandwidthSelection,2,2,range[(rows(range))-3:rows(range)])
          show (RegressionBandwidthSelection,1,1,""|(" "+optionlist[chosen]))
          break
      endsw
;
      mh=regest(x, hopt, K, d)
      setmaskp(x,2,11,2)
      setmaskp(mh,0,0,0)
      setmaskl(mh, (1:rows(mh))', 0, 1, 1)
      show (RegressionBandwidthSelection,2,1, x,mh)
      setgopt(RegressionBandwidthSelection,2,1,"title","Regression Estimate","xlabel","x","ylabel","mh") 
    endif
  endo ; stop
endp