proc(mh, clo, cup) = regcb(x,h,alpha,K,d) 
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      regest regci regxest regxci regxcb 
; -----------------------------------------------------------------
;   Macro        regcb
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel regression, 
;                Nadaraya-Watson estimator, confidence bands
; -----------------------------------------------------------------
;   Description  computes uniform confidence bands with 
;                prespecified confidence level for univariate
;                regression using the Nadaraya-Watson estimator.
;                The computation uses WARPing.
; -----------------------------------------------------------------
;   Reference    Haerdle (1990): Applied Nonparametric Regression
;
;                Haerdle (1991): Smoothing Techniques
; -----------------------------------------------------------------
;   Usage        {mh, clo, cup} = regcb(x {,h {,alpha {,K} {,d}}})
;   Input
;     Parameter  x  
;     Definition   n x 2, the data. In the first column the
;                  independent, in the second column the
;                  dependent variable.
;     Parameter  h 
;     Definition   scalar, bandwidth. If not given, 20% of the
;                  range of x[,1] is used.
;     Parameter  alpha  
;     Definition   confidence level, If not given, 0.05 is used.
;     Parameter  K  
;     Definition   string, kernel function on [-1,1] with
;                  K(-1)=K(1)=0. If not given, the Quartic kernel 
;                  "qua" is used.
;     Parameter  d  
;     Definition   scalar, discretization binwidth. d must be smaller
;                  than h. If not given, the minimum of h/3 and 
;                  (max(x[,1])-min(x[,1]))/100 is used.
;   Output
;     Parameter  mh  
;     Definition   m x 2 matrix, the first column is a grid and the 
;                  second column contains the regression estimate on 
;                  that grid.
;     Parameter  clo  
;     Definition   m x 2 matrix, the first column is a grid and the 
;                  second column contains the lower confidence 
;                  bounds for that grid.
;     Parameter  cup  
;     Definition   m x 2 matrix, the first column is a grid and the 
;                  second column contains the upper confidence 
;                  bounds for that grid.
; -----------------------------------------------------------------
;   Example   library("smoother") 
;             library("plot")                                    
;             x = read("motcyc")
;             {mh, clo, cup} = regcb(x,3)  
;             ;
;             x =setmask( x,"cross")
;             mh=setmask(mh,"line","blue")
;             clo=setmask(clo,"line","blue","thin","dashed")
;             cup=setmask(cup,"line","blue","thin","dashed")
;             plot(x,mh,clo,cup)
; -----------------------------------------------------------------
;   Result    Uniform confidence bands at confidence level 
;             alpha=0.05 are pictured using the bandwidth h=0.3
; -----------------------------------------------------------------
;   Author    Marlene Mueller, 2000/03/28 - 14:15
; -----------------------------------------------------------------
;
  error(cols(x)<>2,"regcb: cols(x) <> 2")
; 
  if (exist(h)==0)
    h=(max(x[,1])-min(x[,1]))/5
  endif
  if (exist(alpha)==0)
   alpha = 0.05                 ; default confidence level is 0.05    
  endif   
  if (exist(K)==0)
    cK=5/7
    c2=1.5
    K="qua"
  endif
  if (exist(K)==1) ; no K but d
    d=K
    cK=5/7
    c2=1.5
    K="qua"
  endif
  if ((exist(K)==2)&&(K!="qua"))
    {dK,cK,mK}=canbw(K)
    {dK,cK,mK}=canbw(K)
    ng=1000
    hg=2/(ng-1)
    g=grid(-1,hg,ng)
    g1=g[1:(rows(g)-1)]
    g2=g[2:rows(g)]
    DK=(_K(g2)-_K(g1))./hg  ; approx. derivative of K
    c1K=sum(DK^2)*hg        ; approx. norm of K'
    c2=c1K/(2*cK)
  endif
;
  if (exist(d)==0)
    d=(max(x[,1])-min(x[,1]))/100
    d=min(d~(h./3),2)
  endif
;
  n = rows(x)
  mx= min(x[,1])
  rx= max(x[,1])-min(x[,1])
;
  x[,1] = (x[,1]-mx)./rx
  d = d./rx
  h = h./rx
  mh = regest(x,h,K,d)
  sh = regest(x[,1]~(x[,2].^2),h,K,d)
  sh = sh[,2] - mh[,2].^2 ; estimate for var(x[,2]|x[,1])
  fh = denest(x[,1],h,K,d)
;
  r  = sqrt(-2*log(h))
  dn = r + 0.5*log( c2/(2*(pi^2)) )/r
  calpha = -log(-log(1-alpha)/2)
;
  mrg = (calpha/r + dn) .* sqrt(cK.*sh./((n*h).*fh[,2]))
  clo = (rx.*mh[,1] + mx)~(mh[,2] - mrg)
  cup = (rx.*mh[,1] + mx)~(mh[,2] + mrg)
  mh  = (rx.*mh[,1] + mx)~(mh[,2])
endp  


