proc (mh)=regxest(x,h,K,v)   
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      regxci regxcb regest regxbwsel lpregxest regxestp 
; -----------------------------------------------------------------
;   Macro        regxest
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel regression, 
;                Nadaraya-Watson estimator
; -----------------------------------------------------------------
;   Description  computes the Nadaraya-Watson estimator for 
;                univariate regression. 
; -----------------------------------------------------------------
;   Notes        This function does an exact computation, i.e.
;                requires O(n^2) operations for estimating the 
;                regression function on all observations. For
;                exploratory purposes, the macro "regest" is
;                recommended, which uses the faster WARPing method.
; -----------------------------------------------------------------
;   Reference    Haerdle (1990): Applied Nonparametric Regression
; -----------------------------------------------------------------
;   Usage        mh = regxest(x {,h {,K} {,v} })
;   Input
;     Parameter  x  
;     Definition   n x 2, the data. In the first column the
;                  independent, in the second column the
;                  dependent variable.
;     Parameter  h 
;     Definition   scalar, bandwidth. If not given, 20% of the
;                  range of x[,1] is used.
;     Parameter  K  
;     Definition   string, kernel function on [-1,1] or Gaussian
;                  kernel "gau". If not given, the Quartic kernel
;                  "qua" is used.
;     Parameter  v  
;     Definition   m x 1, values of the independent variable on 
;                  which to compute the regression. If not given, 
;                  x is used.
;   Output
;     Parameter  mh  
;     Definition   n x 2 or m x 2 matrix, the first column is the 
;                  sorted first column of x or the sorted v, the 
;                  second column contains the regression estimate  
;                  on the values of the first column.
; -----------------------------------------------------------------
;   Example   library("smoother") 
;             library("plot")
;             ;
;             x = 4.*pi.*(uniform(200)-0.5)   ; independent variable
;             m = cos(x)                      ; true function
;             e = uniform(200)-0.5            ; error term             
;             x = x~(m+e)                             
;             ;
;             mh = regxest(x,1)                ; estimate function
;             ;
;             mh = setmask(mh, "line","blue")
;             m  = setmask(sort(x[,1]~m) , "line","black","thin")
;             plot(x,mh,m)                                         
; -----------------------------------------------------------------
;   Result    The Nadaraya-Watson regession estimate (blue) using   
;             Quartic kernel and bandwidth h=1 and the true 
;             regression function (thin black) are pictured.
; -----------------------------------------------------------------
;   Example   library("smoother") 
;             library("plot")
;             ;
;             x = read("motcyc")            ; read motorcycle data
;             mhe = regest(x,3,"epa")       ; estimate function
;             mhu = regest(x,2,"uni")       ; estimate function
;             ;
;             mhe= setmask(mhe,"line","green")
;             mhu= setmask(mhu,"line","red")
;             plot(mhe,mhu)                 ; graph functions
; -----------------------------------------------------------------
;   Result    The Nadaraya-Watson regession estimates using   
;             Epanechnikov kernel (green) and Uniform kernel 
;             (red) are pictured.  
; -----------------------------------------------------------------
;   Author    Marlene Mueller, 990413
; -----------------------------------------------------------------
  n=rows(x)
  error(cols(x)<>2,"regest: cols(x) <> 2")
;
  if (exist(h)==0)
    h=(max(x[,1])-min(x[,1]))/5
  endif
  if (exist(K)==0)
    K="qua"
  endif
  if (exist(K)==1) ; no K but v
    v=K
    K="qua"
  endif
;
  tmp=sort(x)
  x=tmp[,1]
  y=tmp[,2]
  if (exist(v)==0)
    v=x
  endif
;
  if (K!="gau")
    tmp=sker(x,h,K,matrix(n)~y,v)
  else
    tmp=sker(x,5.*h,"ngau",matrix(n)~y,v)
  endif
  mh=v~ (tmp[,2]./tmp[,1])
endp


