proc(rel)=relationcorrcont(x, colname, opt)
; -----------------------------------------------------------------------
; Library      stats
; -----------------------------------------------------------------------
; See_also     relation relationchi2 relationcont relationrank 
;              relationcorr
; -----------------------------------------------------------------------
; Macro        relationcorrcont
; -----------------------------------------------------------------------
; Description  Computes the corrected contingency coefficient for 
;              discrete data.
; ----------------------------------------------------------------------
;   Notes      To compute the corrected contingency coefficient 
;              non-interactively you need to set opt = "automatic". 
;              In the interactive mode
;              you will get a menu sorted after the largest coefficients.
;              If you click on the coefficient you will get the crosstable 
;              of the corresponding variables.
;
; -----------------------------------------------------------------------
; Usage        rel = relationcorrcont (x {, colname {, opt}})
; Input
;   Parameter  x 
;   Definition n x p      variables
;   Parameter  colname
;   Definition n x p      variable names
;   Parameter  opt 
;   Definition q x 1      text vector of optional parameters
; Output
;   Parameter  rel.r      
;   Definition p x p      matrix of corrected contingency coefficients
;   Parameter  rel.pval   
;   Definition p x p      significance level of chi^2 statistics
; -----------------------------------------------------------------------
; Example      ; loads the library stats
;              library("stats")   
;              ; read swiss banknote data
;              ; actually these data are continuous, but the first
;              ; three variables have approx. 20 realizations
;              x = read ("bank2")
;              x = x[,1:3]
;              ; compute the corrected contingency coefficients and the 
;              ; pvalues of the chi^2 statistic automatically
;              colname = string ("X%0.f", 1:cols(x))
;              relationcorrcont (x, colname, "automatic")
; -----------------------------------------------------------------------
; Result   Contents of rel.r
;          [1,]        1  0.81833  0.85195 
;          [2,]  0.81833        1  0.86405 
;          [3,]  0.85195  0.86405        1 
;          Contents of rel.pval
;          [1,]        0  0.20152  0.0024184 
;          [2,]  0.20152        0  3.8681e-11 
;          [3,]  0.0024184  3.8681e-11        0 
; -----------------------------------------------------------------------
; Author       Sigbert Klinke 970820, 981112
; ----------------------------------------------------------------------
  optgiven  = (exist("opt")==2)

  colnamegiven = (exist("colname")==2) 
  if (colnamegiven)
    colnamegiven = (rows(colname)==cols(x))
  endif
  if (!colnamegiven)
    colname = string ("X%.0f", 1:cols(x))
  endif

  p = cols(x)
  i = 0
  r = -1.*matrix(p,p)
  pval = -1.*matrix(p,p)
  while (i<p)
    i = i+1
    j = 0
    while (j<i)
      j = j+1
      {x1, x2, t} = table2(x[,i]~x[,j])
      e = sum(t).*sum(t,2)/rows(x)
      chi2 = (t-e)^2/e
      r[i,j] = sum(sum(chi2,2))
      pval[i,j] = 1-cdfc(r[i,j], (rows(t)-1).*(cols(t)-1)) 
      pval[j,i] = pval[i,j]
      r[i,j] = sqrt(r[i,j]/(r[i,j]+rows(x)))
      cs = min(rows(t)|cols(t))
      r[i,j] = r[i,j].*sqrt(cs/(cs-1)) 
      r[j,i ] = r[i,j] 
    endo
  endo

  automatic = 0
  if (optgiven)
    automatic = sum(opt=="automatic")
  endif
  if (automatic==0)
    vr = vec(r)~grid(#(1,1), #(1,1), #(p,p))
    vr = sort(abs(vr[,1])~vr, -1)
    vr = vr[p+1:rows(vr)]
    vr = paf(vr, (1:rows(vr))%2)
    selhead = "Corr. Contingency"
    vrp = 0
    sel = 1
    while (sum(sel))
      if (vrp==0)
        if (rows(vr)>15) 
          selitem = string ("%7.3f", vr[1:15,2])|"Next"
          selop   = (1:15)|-1
        else
          selitem = string ("%7.3f", vr[1:rows(vr),2])
          selop   = 1:rows(vr)
        endif
      else
        if ((vrp*15+15)<rows(vr))
          selitem = string ("%7.3f", vr[vrp*15+1:vrp*15+15,2])|"Back"|"Next"
          selop   = (vrp*15+1:vrp*15+15)|-2|-1
        else
          selitem = string ("%7.3f", vr[vrp*15+1:rows(vr),2])|"Back"
          selop   = (vrp*15+1:rows(vr))|-2
        endif
      endif
      sel = selectitem(selhead, selitem, "single")
      if (sum(sel))
        ii = paf (1:rows(sel), sel)
        i  = selop[ii]
        if (i==-1)
          if (vrp*15+16<rows(vr))
            vrp = vrp+1
          endif
        endif
        if (i==-2)
          if (vrp)
            vrp = vrp-1
          endif
        endif
        if (i>0)
          rdt = colname[vr[i,3]|vr[i,4]] 
          crosstable(x[,vr[i,3]]~x[,vr[i,4]], rdt)
        endif
      endif
    endo
  endif  
  rel = list (r, pval)
endp
