proc (KP)=rvlm(y,trunclag)
; ----------------------------------------------------------------------
; Library 	  times
; ----------------------------------------------------------------------
;  See_also	  kpss, lo, robwhittle, roblm, lobrob, gph, sclmtest
;----------------------------------------------------------------------
;   Macro	  rvlm
; ----------------------------------------------------------------------
;   Description  Calculation of the rescaled variance test for I(0) against long-memory
;                alternatives. The statistic is the centered kpss statistic based on the 
;                deviation from the mean. The limit distribution of this statistic is a Brownian
;                bridge whose distribution is related to the distribution of the Kolmogorov 
;                statistic. This statistic can also be used for detecting long-memory in ARCH 
;                models.
;                The first argument of the quantlet is the series, the second optional argument 
;                is the vector of truncation lags for the spectral based autocorrelation 
;                consistent estimator of the variance. If this optional argument is not 
;                provided, the default vector of truncation lags used by Kwiatkowski, Phillips,
;                Schmidt and Shin is used. The quantlet returns the order of the truncation lag,
;                the rescaled variance statistic, with the 95% critical value.
; ----------------------------------------------------------------------
;   Usage         k = rvlm(y{,m})
;   Input 
;     Parameter   y 
;     Definition  vector
;     Parameter   m vector
;     Output
;     Parameter   k
;     Definition  vector
; ----------------------------------------------------------------------
;   Example     ;Nonstationarity test on the differenced series dmus58.dat. Since the evaluated
;               ;statistics are below the critical value for I(0), for all the
;               ;truncation orders, we accept the null hypothesis that the series is I(0).
;               ;Thus the level series is I(1)
;               library("times")
;               x = read("dmus58.dat") ; 25477 obs: log FX rates
;               x = tdiff(x[1:2000])
;               k = rvlm(x) ; 
;               k
; ----------------------------------------------------------------------
;   Result
;                 Contents of k
;
;               [1,] "   Order  Statistic  Crit. Value "
;               [2,] "_________________________________"
;               [3,] ""
;               [4,] "      0    0.0383    0.1869"
;               [5,] "      8    0.0428    0.1869"
;               [6,] "     25    0.0411    0.1869"        
;                
;
; ----------------------------------------------------------------------
;   Example     ;Nonstationarity test on the differenced series dmus58.dat. 
;               ;Here, the user is providing the optional vector of truncation lags.
;               ;Since the evaluated statistics are below the critical value for I(0), for all the
;               ;truncation orders, we accept the null hypothesis that the series is I(0).
;               ;Thus the level series is I(1)
;               library("times")
;               x = read("dmus58.dat") ; 25477 obs: log FX rates
;               x = tdiff(x[1:2000])
;               m = #(2,4,6)
;               k = rvlm(x,m) ; 
;               k
; ----------------------------------------------------------------------
;   Result
;                 Contents of k
; 

;                [1,] "   Order  Statistic  Crit. Value "
;                [2,] "_________________________________"
;                [3,] ""
;                [4,] "      2    0.0420    0.1869"
;                [5,] "      4    0.0430    0.1869"
;                [6,] "      6    0.0429    0.1869"
; ----------------------------------------------------------------------
;   Example     ;Long-memory test on the series of absolute returns. 
;               ;Here, the user is providing the optional vector of truncation lags.
;               ;Since the evaluated statistics are over the critical value for I(0), for all the
;               ;truncation orders, a * is displayed after the statistic, and we accept the
;               ;alternative hypothesis that the series of absolute returns display
;               ;long memory.
;               library("times")
;               x = read("dmus58.dat") ; 25477 obs: log FX rates
;               x = abs(tdiff(x[1:2000]))
;               m = #(2,4,6)
;               k = rvlm(x,m) ; 
;               k
; ----------------------------------------------------------------------
;   Result
;                 Contents of k
;                [1,] "   Order  Statistic  Crit. Value "
;                [2,] "_________________________________"
;                [3,] ""
;                [4,] "      2    1.5623 *   0.1869"
;                [5,] "      4    1.4222 *   0.1869"
;                [6,] "      6    1.3237 *   0.1869"

;     
; ----------------------------------------------------------------------
;   Reference
;       Kokoszka, P. and R. Leipus (1998): Rescaled Variance Test of Long-Memory. 
;         Mimeo University of Liverpool, Division of Statistics and Operation Research.
;       Giraitis, L, Kokoszka, P. and R. Leipus (1999): Detection of Long_memory in ARCH
;         Models. Mimeo LSE and University of Liverpool, Division of Statistics and Operation Research.
; ----------------------------------------------------------------------
;  Keywords
;       long memory, tests, nonstationarity
; ----------------------------------------------------------------------
;   Author        Gilles Teyssiere, 990112
; ----------------------------------------------------------------------
  ;;
	n = rows(y)
  opt = 0 ; tests the presence of the optional matrix of truncation lags parameters
  if (exist(trunclag)==1)
   bandwidth = rint(trunclag) 
  else  
   rat1 = (n/100)^(0.25)
   bandwidth = #(0,rint(4*rat1),rint(12*rat1))  
  endif
  nb = rows(bandwidth)   
  ;;
  temps = cumsum(matrix(n))
  ctemps = temps - mean(temps)
  e1 = y - mean(y)
  S1 = sum(cumsum(e1)^2)
	S1 = S1 - (sum(cumsum(e1))^2)/n
  ;;
  KP = matrix(nb)
	i = 1
	while (i <= nb)
  	KP[i] = S1/NeweyWest(e1,bandwidth[i])
		i = i+1
	endo
  
  KP = KP/(n^2)
	CI = matrix(nb)
	i = 1
	while (i <=nb)
		if (KP[i] <= 0.18688); 95% critical value of Brownian Bridge
			CI[i] = 0
		else
			CI[i] = 42
		endif
		i = i+1;
	endo
  critvalue = #(0.18688, 0.18688, 0.18688)
  ;; 
	;; Display of the results
  line1 ="   Order  Statistic  Crit. Value "
  line2 ="_________________________________"
  st1 = string("%7.0f",bandwidth)
  st3 = string("%10.4f",KP)
	st4 = string("%9.4f",critvalue)
  st5= char(matrix(3,2)*32)
	st6 = char(CI)
  st7 = st1+st3+st5+st6+st4
  KP = line1|line2|""|st7
endp
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Newey-West robust variance estimator
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
proc (sigma2) = NeweyWest (y,q)
  ;;    
  sig2 = mean((y-mean(y))^2)
  gcv = acf(y)*var(y) 
  j = 1
  sum1 = 0
  while (j <= q)
    sum1 = sum1 + (1-j/(1+q))*gcv[j+1]
    j = j+1
  endo
  sigma2 = sig2+2*sum1
endp
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; end of macro rvlm
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

