proc (edr, eigen) = sir1 (matYX, nbslices)
;------------------------------------------------------------------------
; Library	metrics
;------------------------------------------------------------------------
; See_also	sssm sir
;------------------------------------------------------------------------
; Macro		sir1
;------------------------------------------------------------------------
; Description   This macro is the taylored Sliced Inverse Regression method
; 		for the "sssm" macro.
; 		Here, the macro "sir1" builds a special slice which contains
; 		the cases such that y=0 (the value "0" for y symbolically
;		indicates a missing value). Then, nbslices (by default,
;		nbslices=5) other slices are made by splitting the range of
;		the non missing y's into slices of nearly equal weight (i.e.
;		such that they contain nearly the same number of cases).
; 		The classical algorithm of S.I.R. is then used with this
;		special slicing.
; ----------------------------------------------------------------------- 
; Usage		{edr, eigen} = sir1 (matYX, nbslices)                  
;------------------------------------------------------------------------
; Input                                                                     
; Parameter	matYX
; Definition  	n x (p+1) matrix containing the value of the dependent
;		variable in its first column, the values of the p
;		explanatory variables in the last p columns,
;		for a sample of size n.
; Parameter	nbslices
; Definition	optional scalar parameter, 
;		number of slices for the non missing y's cases.
;		(By default, it is equal to 5)
; Output                                                                     
; Parameter	edr
; Definition	p x p  matrix which contains the estimated EDR directions
;		(in columns), sorted according to the increasing values
;		of their corresponding eigenvalues.
;
; Parameter     eigen
; Definition	p x 1 matrix which contains the corresponding eigenvalues
;	        sorted according to the increasing values.
;-------------------------------------------------------------------------
; Author	JS, 960916 
; ----------------------------------------------------------------------------
	x=matYX[,2:cols(matYX)]
	y=matYX[,1]            
         n    = rows(x)              ; number of observations                 
         ndim = cols(x)              ; number of dimension                    
       ; calculate the covariance matrix and its inverse root to standardize X
       ; step 1 in original article
         xb   = mean(x)                         ; mean of x                       
         s    = (x'*x - n.*xb'*xb)./(n-1)       ; cov of x                        
         {eval, evec} = eigsm(s)                ; eigendecomposititon of cov   
         si2  = evec*diag(sqrt(1./eval))*evec'  ; compute cov(x)^(-1/2)        
         xt   = (x-xb)*si2                      ; stand. x to mean=0, cov=I    
       ; construct slices in Y space
       ;  step 2 in original article
         data  = sort(y~xt)                     ; sort data with respect to y 
       ; create a special  slice for the missing values of y observations
	missing=0 ; notation of the missing value
	sindex=  y==missing	
	if (sum(sindex)!=0)
		ssxval=paf(xt,sindex)
	endif
	nsindex=matrix(n,1)-sindex
	nsx=paf(xt,nsindex)
	nsy=paf(y,nsindex)
	data1=sort(nsy~nsx)
	n1=rows(data1)
       ; build slices 
        if (exist(nbslices)==0)
		ns=5
        else
		ns=nbslices
	endif
	h      = floor(n1/ns)
        condit = aseq(1, n1, 1)             ; enumber the x values             
        choice = aseq(1, ns, 1).*h         ; take h values in each slice      
        if (h*ns != n1)                     ; if there are remaining values    
           hk = floor((n1-h*ns)/2)
           if (hk>=0)
             choice = hk|choice+hk          ; generate a new first ...         
           endif
           choice = choice|n1                ; ... and last slice               
        endif
        ns = rows(choice)                  ; number of slices                 
       ;  run over all slices, compute estimate for V = Cov(E(z|y))            
       ;  step 3            
         hk   = 0
         v    = matrix(ndim, ndim)*0  ; initialise V matrix                             
	ind  = matrix(n1)             ; index vector of length n; 0 means,         
                                      ;  this values was already choosen             
                                      ;  in a previous slice                         
         d    = ndim            
         ndim = ndim+1                ; sorted x values are in data[,2:ndim]       
         j    = 1
         while (j <= ns)            
           sborder = (condit <= choice[j]) && ind   ; sborder is index            
           p = sum(sborder)                         ; if jth obs. belongs         
           if (p != 0)                              ; current slice               
             ind    = ind - sborder                 ; don't take this values        
                                                    ; in further slices           
             xslice = paf(data1[,2:ndim], sborder)   ; get sliced x-values         
             xmean  = mean(xslice)
             v = v + xmean'*xmean*rows(xslice)      ; compute V                   
             hk = hk+1
           endif
           j = j+1
         endo
	 if (sum(sindex)!=0)
		sxmean=mean(ssxval)
		v=v+sxmean'*sxmean*rows(ssxval)
	 endif
         error(sum(ind) != 0, "Error: sum(ind) elements unused !!")
         v          = (v+v')/2/n              ; for numerical errors 
         {eigen, b} = eigsm(v)                ; step 5, eigenvectors of V  
         edr         = si2*b                   ; calculate e.d.r. direction 
	 o=order(-eigen)
	 edr=edr[,o]
	 eigen=eigen[o]
       endp
