proc(VaR)=VaRest(y,method,opt) 
; ----------------------------------------------------------------------------
; Library        VaR
; ----------------------------------------------------------------------------
;  See_also      VaRopt VaRtimeplot VaRqqplot
; ----------------------------------------------------------------------------
;   Macro        VaRest
; ----------------------------------------------------------------------------
;   Description  VaRest estimates the value at risk (VaR).
; ----------------------------------------------------------------------------
;   Usage        VaR = VaRest(y{,method}{,opt})
;   Input
;     Parameter   y  
;     Definition        n x d  matrix, the returns of d assets.
;     Parameter   method
;     Definition        method for VaR, one of 
;                       "RMA" (rectangular moving average), 
;                       "EMA" (exponential moving average), 
;                       "MAD" (mean absolute deviation), 
;                       "EDF" (empirical distribution function). 
;                       The default method is "RMA".
;     Parameter   opt
;     Definition        optional, a list with optional input. The function
;                       "VaRopt" can be used to set up this parameter.
;                       The order of the list elements is not important.
;                       Parameters which are not given are replaced by 
;                       defaults (see below).
;     Parameter   opt.h
;     Definition        positive integer, window width. 
;                       If not given, set to 250.
;     Parameter   opt.lam
;     Definition        positive scalar, parameter for EMA method.
;                       If not given, set to 0.96.
;     Parameter   opt.dist
;     Definition        integer, distribution. The default is 0 for
;                       normal, a positive integer denotes the degrees 
;                       of freedom when using a t-distribution.
;     Parameter   opt.alpha
;     Definition        scalar in (0,1), significance level. The 
;                       default is 0.01.
;     Parameter   opt.w
;     Definition        scalar, 1 x d or n x d, weights for assets. 
;                       If not given, set to 1.
;     Parameter   opt.bw
;     Definition        scalar, bandwidth for method "KDQ" for
;                       quantiles from a kernel density estimator.
;                       If not given, chosen by Silverman's rule
;                       of thumb.
;   Output
;     Parameter   VaR
;     Definition        (n-h) x 2  vector, the VaR for observations
;                       h+1 to n.
; ----------------------------------------------------------------------------
;   Example   library("VaR")
;             x=read("kupfer")       ; time series
;             x=x[1:1001]  
;             y=diff(log(x))         ; returns
;             VaR=VaRest(y)          ; VaR 
;             opt=VaRopt("exceed","blue"|"fillcircle"|"small")
;             VaRtimeplot(y,VaR,opt) ; VaR time plot
; ----------------------------------------------------------------------------
;   Result    The VaR forecasts (using method RMA) are plotted 
;             together with the portfolio returns. Returns which
;             exceed the VaR are marked in blue.
; ----------------------------------------------------------------------------
;   Example   library("VaR")
;             x=read("kupfer")       ; time series
;             x=x[1:1001]  
;             y=diff(log(x))         ; returns
;             VaRrma=VaRest(y,"RMA") ; VaR using RMA 
;             VaRema=VaRest(y,"EMA") ; VaR using EMA
;             VaRmad=VaRest(y,"MAD") ; VaR using MAD
;             opt=VaRopt("color","red"|"blue"|"green")
;             VaRtimeplot(y,VaRrma,VaRema,VaRmad,opt)  
; ----------------------------------------------------------------------------
;   Result    The VaR forecasts (using methods RMA, EMA and MAD) are plotted 
;             together with the portfolio returns.
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 2000/07/05
; ----------------------------------------------------------------------------
;
; check initial errors
;
  bad=(rows(dim(y))>2)
  error(bad>0,"y must be vector or matrix")
;
  n=rows(y)
  d=cols(y)
;
; set the defaults
;
  if (exist(method)==9)
    opt=method
    method="RMA"
  endif
;
  allmethods="RMA"|"MAD"|"EMA"|"EDF"|"KDQ"
  if (exist(method)<=0)
    method="RMA"
  else
    notgood=(sum(method==allmethods)==0) 
    error(notgood>0, "Method string is invalid")
  endif
;
  h=250
  lam=0.96
  dist=0
  alpha=0.01
  w=1
  bw=0
;
; now check which optional values have been given
;
  if (exist(opt)>0)
    if (comp(opt,"h")>0)
      opt.h=opt.h[1]
      notgood=(opt.h>=n-1)||(opt.h<2)
      error(notgood>0, "VaRest: Window width h is unusable")
      if (notgood==0)
        h=opt.h
      endif
    endif
    if (comp(opt,"lam")>0)
      opt.lam=opt.lam[1]
      notgood=(opt.lam>=1.0)||(opt.lam<=0)
      error(notgood>0, "VaRest: Parameter lam is unusable")
      if (notgood==0)
        lam=opt.lam
      endif
    endif
    if (comp(opt,"dist")>0)
      opt.dist=round(opt.dist[1])
      notgood=(opt.dist<0)
      error(notgood>0, "VaRest: Distribution parameter dist is unusable")
      if (notgood==0)
        dist=opt.dist
      endif
    endif
    if (comp(opt,"alpha")>0)
      opt.alpha=opt.alpha[1]
      notgood=(opt.alpha<0)
      error(notgood>0, "VaRest: Significance level alpha is unusable")
      if (notgood==0)
        alpha=opt.alpha
      endif
    endif
    if (comp(opt,"w")>0)
      notgood=((cols(opt.w)!=d)&&(cols(opt.w!=1)))
      notgood=notgood || ((rows(opt.w)!=n)&&(rows(opt.w!=1)))
      if (notgood==0)
        w=opt.w
      endif
    endif
    if (comp(opt,"bw")>0)
      opt.bw=opt.bw[1]
      notgood=(opt.bw<=0)
      error(notgood>0, "VaRest: Bandwidth bw is unusable")
      if (notgood==0)
        bw=opt.bw
      endif
    endif
  endif
  w=w.*matrix(1,d)
;
  switch
    case (method=="RMA")
      sigh=matrix(n-h)-1
      tmp=cumsum(y.*reshape(y,n|1|d))
      wtr=reshape(w,rows(w)|1|d)
      tmp=(tmp[(h+1):n]-tmp[1:(n-h)])./h
      sigh=sqrt(sum(sum(wtr.*tmp,3).*w,2))
      break
    case (method=="EMA")
      sigh=matrix(n-h)-1
      wtr=reshape(w,rows(w)|1|d)
      j=h
      while (j<n)
        j=j+1
        tmp=(lam^grid(h-1,-1,h)).*y[(j-h):(j-1)]
        tmp=sum(tmp.*reshape(tmp,h|1|d))
        sigh[j-h]=sqrt(sum(sum(wtr.*tmp,3).*w,2).*(1-lam))
      endo
      break
    case (method=="MAD")
      sigh=matrix(n-h)-1
      tmp=cumsum(abs(y))
      tmp=tmp[(h+1):n]-tmp[1:(n-h)]
      sigh=((tmp./h)./0.798)           ; robust standard errors
      if (d>1)
        tmp=cumsum(y.*reshape(y,n|1|d))
        wtr=reshape(w,rows(w)|1|d)
        tmp=(tmp[(h+1):n]-tmp[1:(n-h)])./h            ; covariance
        sigd=sqrt(tmp[,1,1])
        j=1
        while (j<d)
          j=j+1
          sigd=sigd~sqrt(tmp[,j,j])
        endo
        tmp=tmp./(sigd.*reshape(sigd,rows(sigd)|1|d)) ; correlation
        tmp=tmp.*(sigh.*reshape(sigh,rows(sigh)|1|d)) ; rob. covariance
        wtr=reshape(w,rows(w)|1|d)
        sigh=sqrt(sum(sum(wtr.*tmp,3).*w,2))
      endif
      break
    case (method=="EDF")
      Y=sum(w.*y,2);  portfolio!
      qf=matrix(n-h,2)-1
      j=h
      while (j<n)
        j=j+1
        qf[j-h]=quantile(Y[(j-h):(j-1)],alpha|(1-alpha))
      endo
      break
    case (method=="KDQ")
      Y=sum(w.*y,2);  portfolio!
      qf=matrix(n-h,2)-1
      j=h
      while (j<n)
        j=j+1
        if (bw>0)
          kde=denest(Y[(j-h):(j-1)],bw)
        else
          kde=denest(Y[(j-h):(j-1)])
        endif
;        Yr=(max(Y)-min(Y))./100
;        Yg=grid(min(Y)+0.5*Yr,Yr,100)
;        kde=denxest(Y[(j-h):(j-1)],denrot(Y[(j-h):(j-1)]),"qua",Yg)
        binwidth=kde[2,1]-kde[1,1]
        kde=((kde[1,1]-binwidth)~0)|kde|((kde[rows(kde),1]+binwidth)~0)
        ng=rows(kde)
        kde[,2]=kde[,2]./(0.5*binwidth.*(sum(kde[2:ng,2])+sum(kde[1:(ng-1),2])))
        cdf=0|(0.5*binwidth.*(cumsum(kde[2:ng,2])+cumsum(kde[1:(ng-1),2])))
        if (alpha<(1-alpha))
          al=alpha
          ar=1-alpha
        else
          al=1-alpha
          ar=alpha
        endif
        il=maxind(cdf>al)  ; upper interval index of left quantile
        ir=maxind(cdf>ar)  ; upper interval index of right quantile
        il=il+(il==1)      ; make il>1 ... just to be sure ;-)
        F=cdf[(il-1):il]
        X=kde[(il-1):il,1]
        qf[j-h,1]=X[1]+(X[2]-X[1]).*(al-F[1])./(F[2]-F[1])
        F=cdf[(ir-1):ir]
        X=kde[(ir-1):ir,1]
        qf[j-h,2]=X[1]+(X[2]-X[1]).*(ar-F[1])./(F[2]-F[1])
        if ((1-alpha)<alpha)
          qf[j-h,]=qf[j-h,2:1]
        endif
      endo
      break
    default;
      break
  endsw
;
  if ((method!="EDF")&&(method!="KDQ"))
    if (dist==0)
      qf=qfn(alpha)
    else
      sigh=sigh./sqrt(dist/(dist-2))
      qf=qft(alpha,dist)
    endif
;
    VaR=qf.*sigh
    VaR=VaR~(-VaR)
  else
    VaR=qf
  endif
endp 




