proc(VaR)=VaRpred(y,method,opt) 
; ----------------------------------------------------------------------------
; Library        VaR
; ----------------------------------------------------------------------------
;  See_also      VaRest VaRopt VaRtimeplot VaRqqplot
; ----------------------------------------------------------------------------
;   Macro        VaRpred
; ----------------------------------------------------------------------------
;   Description  VaRpred predicts the value at risk (VaR).
; ----------------------------------------------------------------------------
;   Usage        VaR = VaRpred(y{,method}{,opt})
;   Input
;     Parameter   y  
;     Definition        n x d  matrix, the returns of d assets.
;     Parameter   method
;     Definition        method for VaR, one of 
;                       "BETA" (beta factor model), 
;                       "BETACOR" (corrected beta factor model), 
;                       "PRINC" (principal components), 
;                       "PRINCW" (weighted principal components). 
;                       "SUMS" (sums of uncorrelated variables). 
;                       "COMPLETE" (no reduction of variance matrix). 
;                       default is the "COMPLETE" option, the beta 
;                       factor models assume that the first column 
;                       in the data matrix is the index.
;     Parameter   opt
;     Definition        optional, a list with optional input. The macro
;                       "VaRopt" can be used to set up this parameter.
;                       The order of the list elements is not important.
;                       Parameters which are not given are replaced by 
;                       defaults (see below).
;     Parameter   opt.alpha
;     Definition        scalar in (0,1), significance level. The 
;                       default is 0.01.
;     Parameter   opt.w
;     Definition        scalar, 1 x d or n+1 x d, weights for assets. 
;                       If not given, set to 1, the first coordinate
;                       is not used in computation for beta factor models.
;   Output
;     Parameter   VaR
;     Definition        1 x 2  vector, the VaR for observation n+1.
; ----------------------------------------------------------------------------
;   Example   
; ----------------------------------------------------------------------------
;   Result    
; ----------------------------------------------------------------------------
;   Author    Zdenek Hlavka, 2000/06/30 
; ----------------------------------------------------------------------------
;
; check initial errors
;
  bad=(rows(dim(y))>2)
  error(bad>0,"y must be vector or matrix")
;
  d=cols(y)
  n=rows(y)
;
; set the defaults
;
  if (exist(method)==9)
    opt=method
    method="COMPLETE"
  endif
;
  allmethods="BETA"|"BETACOR"|"PRINC"|"PRINCW"|"SUMS"|"COMPLETE"
  if (exist(method)<=0)
    method="COMPLETE"
  else
    notgood=(sum(method==allmethods)==0) 
    error(notgood>0, "Method string is invalid")
  endif
;
  h=rows(y)
  alpha=0.01
  weight=matrix(1,d)
;
; now check which optional values have been given
;
  if (exist(opt)>0)
    if (comp(opt,"alpha")>0)
      opt.alpha=opt.alpha[1]
      notgood=(opt.alpha<0)
      error(notgood>0, "VaRest: Significance level alpha is unusable")
      if (notgood==0)
        alpha=opt.alpha
      endif
    endif
    if (comp(opt,"w")>0)
      if (sum(method==("BETA"|"BETACOR"))>0)
       notgood=(cols(opt.w)!=(d-1))
      else
       notgood=(cols(opt.w)!=(d))
      endif
      notgood=notgood || ((rows(opt.w)!=(n+1))&&(rows(opt.w)!=1))
      if (notgood==0)
        if (rows(opt.w)==n+1)
          weight=opt.w[n+1,]
        else
          weight=opt.w
        endif
      else
        error(1,"wrong dimension of weights!")
      endif
    endif
  endif
;  weight=weight.*matrix(1,d)
  w=diag(weight');
;
  switch
    case (method=="BETA")
;  
;  assume that 1-st column is the index
;      
;      weight=weight[1,2:d]
      ind=y[,1]~matrix(n,1)
      beta=inv(ind'*ind)*ind'*y[,2:d];
      sm=y[,1]'*y[,1]/n;
      sig=weight*beta[1,]'*beta[1,]*sm*weight'; 
      break
    case (method=="BETACOR")
;  
;  assume that 1-st column is the index
;      
      weight=weight[1,2:d]
      ind=y[,1]~matrix(n,1)
      beta=inv(ind'*ind)*ind'*y[,2:d]
      sm=y[,1]'*y[,1]/n
      d=xdiag((y[,2:d]-y[,1]*beta[1,])'*(y[,2:d]-y[,1]*beta[1,])/n)
      dd=diag(d)
      sig=weight*(beta[1,]'*beta[1,]*sm+dd)*weight'
      break
    case (method=="PRINC")
      {eval2,evec2}=eisrs(cov(y))
      evec2=paf(evec2',(((cumsum(eval2/sum(eval2))))).>0.2)
      eval2=eval2[d-rows(evec2)+1:d]
      d2=diag(eval2)
      sig=weight*(evec2')*d2*evec2*(weight')
      break 
    case (method=="PRINCW")
      {eval2,evec2}=eisrs(cov(y*w))
      evec2=paf(evec2',(((cumsum(eval2/sum(eval2))))).>0.2)
      eval2=eval2[d-rows(evec2)+1:d]
      d2=diag(eval2)
      sig=matrix(1,d)*(evec2')*d2*evec2*matrix(d,1)
      break
    case (method=="SUMS")
      xx=paf(y',(1-(weight==0))')'
      weight2=paf(weight',1-(weight'==0))
      dw2=rows(weight2)
      w2=diag(weight2)
      trnsf=matrix(2,1)
      while (1-(rows(trnsf)==cols(trnsf)))
        vc=corr(xx)
        trnsf=VaRauxsums(vc,w2,2/sqrt(n))
        pompr=rows(trnsf)
        xx=xx*trnsf'
        w2=unit(pompr)
      endo
      ww=matrix(pompr,1)
      sig=ww'*cov(xx)*ww
      break
    case (method=="COMPLETE")
      sig=weight*cov(y)*weight'
      break
  endsw
;
 qf=qfn(alpha)
;
 VaR=qf.*sqrt(sig)
 VaR=VaR~(-VaR)
endp 
