 proc(delta,dvar)=adeind(x,y,d,m)
; -----------------------------------------------------------------
; Library        metrics
; -----------------------------------------------------------------
;  See_also      adeslp dwade trimper wtsder
; -----------------------------------------------------------------
;   Macro        adeind
; -----------------------------------------------------------------
;   Keywords     single index model, semiparametric estimation, 
;		 average derivative
; -----------------------------------------------------------------
;   Description  indirect average derivative estimation using binning
; -----------------------------------------------------------------
;   Reference    Haerdle and Stoker (1989), "Investigating Smooth
;		 Multiple Regression by the Method of Average Derivatives", 
;                Journal of the American Statistical Association, 84, 986-995
; -----------------------------------------------------------------
;   Usage        {delta,dvar} = adeind(x,y,d,m)    
;   Input
;     Parameter   x  
;     Definition      n x p matrix , the observed explanatory variable    
;     Parameter   y 
;     Definition      n x 1 matrix , the observed response variable  
;     Parameter   d   
;     Definition      p x 1 vector or scalar , the binwidth or the grid
;     Parameter   m
;     Definition      p x 1 vector or scalar , the bandwidth to be used
;                                    during estimation of the scores 
;   Output                                                           
;     Parameter   delta 
;     Definition      p x 1 vector , the ADE estimate
;     Parameter   dvar
;     Definition      p x p matrix , the estimated asymptotic 
;                                    covariance matrix of delta  
; -----------------------------------------------------------------
;   Example   library("metrics")
;	      randomize(0)
;             n   = 100
;             x   = normal(n,3)
;             z   = 0.2*x[,1] - 0.7*x[,2] + x[,3]
;             eps = normal(n,1) * sqrt(0.5)
;             y   = 2 * z^3 + eps
;             d   = 0.2
;             m   = 5
;             {delta,dvar} = adeind(x,y,d,m)
;             delta
;             dvar
; -----------------------------------------------------------------
;   Result    the indirect regression estimator for average derivative
;             and its asymtotic covariance matrix 
;             as described by Haerdle and Stoker, JASA (1989)
;             and Turlach, Discussion Paper (1993)
; -----------------------------------------------------------------
;   Author    Turlach (in XploRe3),
;             Sperlich & Stockmeyer 960806
; -----------------------------------------------------------------
  error(rows(x)!=rows(y),"x and y have different number of rows")
  error(cols(y)!=1 ,"y has more than one column")
  p=cols(x)
  n = rows(x)
  if ( rows(m) .< p ) 
    m = matrix(p)*m
  endif 
  if ( rows(d) .< p ) 
    d = matrix(p)*d 
  endif
  wx=matrix(p)*0 
  {xb,yb} = bindata(x,d,wx,y)
  sel=(yb[,1].>0)
  xb=paf(xb,sel)
  yb=paf(yb,sel)
  vect= grid(1,1,cols(xb~yb)) 
  xb=sort(xb~yb,vect)
  fb=xb[,p+1]
  rb=xb[,p+2]
  xb=xb[,1:p]
  wts=symweigh(wx,(1./m),(m.+1),"qua")
  wx=grid(wx,matrix(p),(m.+1))
  {xc,yc,or}=conv(xb,fb,wx,wts)
  vect = grid(1,1,cols(xc~yc~or))
  xc2 = sort(xc~yc~or,vect)
   xc = xc2[,1:p]   
   yc = xc2[,p+1]
   or = xc2[,p+2]
  fh = paf(yc,or)
  xt = trimper(fh~fb~rb~xb,5)
  fh=xt[,1]
  fb=xt[,2]
  rb=xt[,3]
  xb=xt[,4:(p+3)]
  fhder=matrix(rows(xb),p)*0
  i    = 1
  sym  = 1
  WHILE (i<= p)
    wts  = wtsder(p,m,i)
    {xc,yc,or} = conv(xb,fb,wx,wts,sym)
   xc2 = sort(xc~yc~or,vect) 
   xc = xc2[,1:p]   
   yc = xc2[,p+1]
   or = xc2[,p+2]
    fht  = paf(yc,or)
    fht  = fht / d[i,1]
    fhder[,i] = fht
    sym  = sym*2
    i    = i+1
  ENDO
  fhder = -fhder ./fh
  delta = fhder'*rb /n
;
; Calculate variance
;
  r     = matrix(rows(xb),p)*0
  i     = 1
  sym   = 1
  WHILE (i<= p)
    wts  = wtsder(p,m,i)
    {xc,yc,or} = conv(xb,rb,wx,wts,sym)
    xc2 = sort(xc~yc~or,vect) 
    xc = xc2[,1:p]   
    yc = xc2[,p+1]
    or = xc2[,p+2]
    fht  = paf(yc,or)
    fht  = fht / d[i,1]
    r[,i] = fht
    sym  = sym*2
    i    = i+1
  ENDO
  r    = r./fh + (rb./fb).*fhder
  xbar = sum(r.*fb,1)/n
  dvar = r'*(r.*fb)/n - xbar.*xbar'
endp