proc (archt) = archtest(y,arg1,arg2)
; ----------------------------------------------------------------------
; Library 	  times
; ----------------------------------------------------------------------
;  See_also	  annarchtest
;----------------------------------------------------------------------
;   Macro	  archtest
; ----------------------------------------------------------------------
;   Description  This macro calculates either the Lagrange Multiplier (LM)
;                form or the R squared (TR2) form of Engle's ARCH test. 
;                The first argument of the function is the vector of residuals,
;                the second optional argument is the lag order of the test. This                ;  		 second argument may be either a scalar, or a vector. In the 
;                later case, the test is evaluated for all the order 
;                components of the vector. If this second optional argument is
;                not provided, the default lag orders are 2, 3, 4, and 5. 
;                The third optional argument is the form of the test. This 
;                argument is a string, which can be either "LM" or "TR2".
;                The default third argument is "LM", i,e., the LM form.
;                The macro returns in the first column the order of the test, 
;                in the second column the value of the test, in the third column
;                the 95% critical value of the null hypothesis for that order, 
;                and in the fourth column the P-value of the test.
; ----------------------------------------------------------------------
;   Usage         h = archtest(y {,lags{,testform}})
;   Input 
;     Parameter   y 
;     Definition  vector
;     Parameter   lags
;     Definition  vector
;     Parameter   testform
;     Definition  string
;     Output
;     Parameter   h
;     Definition  vector of strings
; ----------------------------------------------------------------------
;   Example   
;               library("times")
;               randomize(0)
;               y = genarch(0.2|0.1,0.5,200)
;               h = archtest(y)
;               h
;  
; ----------------------------------------------------------------------
;   Result       As no optional argument is provided, the Lagrange Multiplier 
;                form of the statistic is evaluated for the default lag 
;                orders 2,3,4, and 5.
;               
;                Contents of h
;
;                [1,] Lag order  Statistic  95% Critical Value  P-Value  
;                [2,] __________________________________________________ 
;                [3,]  
;                [4,]      2      3.36892         5.97378       0.18551 
;                [5,]      3     16.40116         7.80251       0.00094 
;                [6,]      4     16.79456         9.47844       0.00212 
;                [7,]      5     17.53251        11.06309       0.00359   
;-----------------------------------------------------------------------
;   Example     ;Calculation of the TR2 form of the ARCH test for the default
;               ;lag orders 2,3,4, and 5.
;               library("times")
;               randomize(0)
;               y = genarch(0.2|0.1,0.5,200)
;               h = archtest(y,"TR2")
;               h
; ----------------------------------------------------------------------
;   Result       
;                Contents of h
;
;                [1,] Lag order  Statistic  95% Critical Value  P-Value  
;                [2,] __________________________________________________ 
;                [3,]  
;                [4,]      2      2.53583         5.97378       0.28139 
;                [5,]      3     12.14567         7.80251       0.00690 
;                [6,]      4     12.42151         9.47844       0.01448 
;                [7,]      5     13.02704        11.06309       0.02313 
;-----------------------------------------------------------------------
;   Example    Calculation of the TR2 form of the ARCH test for the lag
;              orders 3,5,7,8 
;               library("times")
;               randomize(0)
;               y = genarch(0.2|0.1,0.5,200)
;               k = #(3,5,7,8)
;               h = archtest(y,k,"TR2")
;               h
; ----------------------------------------------------------------------
;   Result      
;                Contents of h
;
;                [1,] Lag order  Statistic  95% Critical Value  P-Value  
;                [2,] __________________________________________________ 
;                [3,]  
;                [4,]      3     12.14567         7.80251       0.00690 
;                [5,]      5     13.02704        11.06309       0.02313 
;                [6,]      7     14.13459        14.06200       0.04884 
;                [7,]      8     13.80119        15.50292       0.08710 
; ----------------------------------------------------------------------
;   Example    Calculation of the LM form of the ARCH test for the lag
;              orders 3,5,7,8 
;               library("times")
;               randomize(0)
;               y = genarch(0.2|0.1,0.5,200)
;               k = #(3,5,7,8)
;               h = archtest(y,k)
;               ; h = archtest(y,k,"LM") ; this is an equivalent instruction 
;               h
;-----------------------------------------------------------------------
;   Result   
;             Contents of h
;
;                [1,] Lag order  Statistic  95% Critical Value  P-Value  
;                [2,] __________________________________________________ 
;                [3,]  
;                [4,]      3     16.40116         7.80251       0.00094 
;                [5,]      5     17.53251        11.06309       0.00359 
;                [6,]      7     19.22952        14.06200       0.00750 
;                [7,]      8     18.65397        15.50292       0.01682 
;-----------------------------------------------------------------------
;   Reference
;    Engle, R.F. (1982): Autoregressive Conditional Heteroskedasticity, 
;		with Estimates of the Variance of UK Inflation.
;      		Econometrica, vol 50, pp 987-1006.
; ----------------------------------------------------------------------
;   Keywords
;      ARCH tests, tests
; ----------------------------------------------------------------------
;   Author        Gilles Teyssiere, 980531
; ----------------------------------------------------------------------
  ; default parameters
  lags = #(2,3,4,5);  ARCH test lag orders  
  ttest = 0       ;  the LM form of the test
  targ1 = exist(arg1)
  targ2 = exist(arg2)
  if (targ1 != 0)
    if (targ1 == 1)     
      lags = arg1
      if (targ2 == 2)         
        if (arg2 == "TR2")
          ttest = 1
        endif
      endif
    else
      if (targ1 == 2)
        if (arg1 == "TR2")
          ttest = 1
        endif
      endif
    endif
  endif
  ;;
  n = rows(y)
  ssy = y^2
  nblags = rows(lags)
  archt = matrix(nblags)
  i = 1
  while (i <= nblags)
    rtx = lags[i]
    ty = ssy[rtx+1:n,]
    ty = ty/mean(ty) -1 
    tx = matrix(n-rtx)~ssy[1:n-rtx,] 
    j = 2
    while (j <= rtx)
      idx = n-rtx+j-1
      tx = tx~ssy[j:idx,]
      j = j+1
    endo     
    txy = tx'*ty  
    if (ttest == 0)
      archt[i] = (txy'*inv(tx'*tx)*txy)/2; LM FORM
    else
      archt[i] = (n-rtx)*txy'*inv(tx'*tx)*txy/(ty'*ty); R SQUARE FORM
    endif
    i = i+1  
  endo
  CritOrder = matrix(nblags)*0.95
  ;archt = lags~archt~qfc(CritOrder,lags)~(1-cdfc(archt,lags))
  ; Output display
  line1 ="Lag order  Statistic  95% Critical Value  P-Value "
  line2 ="__________________________________________________"
  st1 = string("%6.0f",lags)
  st2 = string("%13.5f",archt)
  st3 = string("%16.5f",qfc(CritOrder,lags))
  st4 = string("%14.5f",(1-cdfc(archt,lags)))
  st5 = st1+st2+st3+st4
  archt = line1|line2|""|st5
endp  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; end of macro archtest
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 
  
