proc(hrot)=denrot(x, K, opt)
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      denbwsel denbwcrit
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel density estimation, 
;                bandwidth selection, Silverman's rule of thumb
; -----------------------------------------------------------------
;   Macro        denrot
; -----------------------------------------------------------------
;   Description  determines a rule-of-thumb bandwidth for univariate 
;                density estimation according to Silverman.
; -----------------------------------------------------------------
;   Reference    Silverman (1986): Density Estimation for Statistics
;                and Data Analysis
; -----------------------------------------------------------------
;   Usage        hrot = denrot(x {,K {,opt}})
;   Input
;     Parameter  x  
;     Definition   n x 1 vector, the data.  
;     Parameter  K  
;     Definition   string, kernel function on [-1,1]^p. If not given,
;                  the Quartic kernel "qua" is used.
;     Parameter  opt  
;     Definition   string, if opt equals "robust", a robust estimate 
;                  for the standard deviation of the data is used.
;   Output
;     Parameter  hrot  
;     Definition   scalar, rule-of-thumb bandwidth.
; -----------------------------------------------------------------
;   Example   library("smoother")
;             x=read("buffa")
;             hrot=denrot(x)
;             hrot
; -----------------------------------------------------------------
;   Result    Contents of hrot
;             [1,]   28.772 
; -----------------------------------------------------------------
;   Example   library("smoother")
;             x=read("buffa")
;             hrot=denrot(x,"qua","robust")
;             hrot
; -----------------------------------------------------------------
;   Result    Contents of hrot
;             [1,]   31.202 
;-----------------------------------------------------------------
;   Author    Marlene Mueller, 990413
; ----------------------------------------------------------------- 
  error(cols(x)<>1,"denrot: cols(x) <> 1")
;
  if (exist(K)==0)
    K="qua"
  endif
  if (exist(opt)==0)
    opt="null"
  endif
;
  {dK,cK,mK}=canbw(K)
  {dG,cG,mG}=canbw("gau")
  fac=(4/3)^(0.2) .* dK./dG
;
  n = rows(x)
;
  if (opt=="robust")
    lambda = diff(quantile(x,0.25|0.75))
    qn     = diff(qfn(0.25|0.75))
    hrot  = fac .* (lambda/qn) .* n^(-0.2)
  else
    hrot  = fac .* sqrt(var(x)) .* (n^(-0.2))
  endif
endp   
