proc(beta)=eivlinearinstrvec(x,w,y)
; -----------------------------------------------------------------
;    Library	eiv
; -----------------------------------------------------------------
;    See_also  eivlinearinstr
; -----------------------------------------------------------------
;    Macro	eivlinearinstrvec
; -----------------------------------------------------------------
;    Description    eivlinearinstrvec handle vector-explanatory variable
;		    model, which extends the results given by eivlinearinstr.
;		    The calculating results are based on moment methods.
; ----------------------------------------------------------------------------
;   Reference	 Fuller W. A. "Measurement Error Models", Wiley and Sons 
;              1987, page 148-151.
; -----------------------------------------------------------------
;    Link	  ../tutorials/eivstart.html Tutorial: EIV in XploRe
; -----------------------------------------------------------------
;    Usage    beta = eivlinearinstrvec(x,w,y)
;    Input
;      Parameter  x
;      Definition   n x 1 matrix, the design variables
;      Parameter  w
;      Definition   n x 1 matrix, the instrumental variables
;      Parameter  y
;      Definition   n x 1 matrix, the response
;    Output
;      Parameter  beta
;      Definition   vector, the estimate
; -----------------------------------------------------------------
;  Example library("xplore")
;	   library("eiv")
;	   n=100
;	   randomize(n)
;	   x=floor(uniform(n,3)*6+#(4,5,5)')
;	   w=floor(uniform(n,4)*8+#(3,3,2,2)')
;	   y=floor(uniform(n)*9+2)
;	   gest=eivlinearinstrvec(x,w,y)
;	   gest
; -----------------------------------------------------------------
;    Result  Contents of gest
;            [1,]  0.19413 
;            [2,]  0.24876 
;            [3,]  0.37562 
; -----------------------------------------------------------------
;   Author    Hua Liang,  970730
; ----------------------------------------------------------
n=rows(w)
     s1=cols(x)
    if(s1==cols(w))
      beta=inv(w'*w)*(w'*y)
    else
     z=y~x
     Saa=inv(n-s1)*(z'*z-z'*w*inv(w'*w)*w'*z)
     Saa11=Saa[1:s1,1:s1]
     Saa22=Saa[s1+1,s1+1]
     Saa21=Saa[1:s1,1+s1]
     {eval,evec}=eigsm(inv(z'*z)*Saa)
     eval=sort(eval,1)
     gamma=inv(eval[s1+1])
     beta=inv(x'*x-gamma*Saa22)*(x'*y-gamma*Saa21)
    endif
endp
