proc(X,Ind)=epscontnorm(T,eps,mid,Cid,mcont,Ccont,DirNorm)

; -----------------------------------------------------------------
; Library        kalman
; -----------------------------------------------------------------
;  See_also     kemitor
; -----------------------------------------------------------------
;   Macro        epscontnorm
; -----------------------------------------------------------------
;   Description  Produces T i.i.d. Variates from an eps-contamination
;                     Model P= (1-eps) N(mid,Cid) + eps K 
;
;                     with K=N(mcont,Ccont)          if DirNorm ==0
;
;                     with K=dirac(mcont)               if DirNorm == -1
; 
;                     with K=dirac( +/- mcont)         if DirNorm == 1
;
; -----------------------------------------------------------------
;   Keywords     state-space models
; -----------------------------------------------------------------
;   Usage         {X,Ind}= epscontnorm(T,eps,mid,Cid,mcont,Ccont,DirNorm)    
;   Input
;     Parameter   T   
;     Definition      number of Variates to be generated
;     Parameter   eps (contamination radius)
;     Definition      numeric in [0,1)
;     Parameter   mid    (ideal mean)
;     Definition      n x 1 vector 
;     Parameter   Cid     (ideal Covariance)
;     Definition      n x n matrix
;     Parameter   mcont    (contaminating mean)
;     Definition      n x 1 vector 
;     Parameter   Ccont     (contaminating Covariance)
;     Definition      n x n matrix
;     Parameter   DirNorm  (decides which type of contaminating distribution K is taken)    
;     Definition   K=N(mcont,Ccont)          if DirNorm ==0
;                     K=dirac(mcont)               if DirNorm == -1
;                     K=dirac( +/- mcont)         if DirNorm == 1
;   Output                                                           
;     Parameter   X  
;     Definition      T x n matrix of generated Variates, 
;                     T is the number of generated Variates,
;                     n is the dimension of generated Variates
;     Parameter   Ind
;     Definition      T Indicator of contamination ( 0 = no cont. 1 = cont.)
; -----------------------------------------------------------------
;   Example   library("kalman")
;                   library("plot")
;                   randomize(0)
;                   T = 100
;                   eps = 0.1
;                   mid=#(0,0)
;                   Cid = #(2,1)~#(1,1)
;                   mcont=#(3,3)
;                   Ccont = #(3,0)~#(0,0.2)
;                   erg=epscontnorm(T,eps,mid,Cid,mcont,Ccont,0)
;                   color=2*erg.Ind+2 ; red (4)  if clipped green (2) else
;                   data=erg.X
;                   setmaskp(data,color, 3, 8) ; rest :default
;                   disp = createdisplay(1,1)
;                   show(disp,1,1,data)
; -----------------------------------------------------------------
;   Result    Generates 100 observations of 
;                 (1-eps) N(0,Cid)+eps N(mcont,Ccont)
;                 and plots the contaminated data in red, the rest in green
; -----------------------------------------------------------------
;   Author    P.Ruckdeschel  991020
; -----------------------------------------------------------------

; dimensions check
  
  Cids=svd(Cid)
  Cconts=svd(Ccont)
  ;
  error(sum (Cids.l<0) > 0 , "Matrix Cid is not positive-semi-definite.")
  error(sum (Cconts.l<0) > 0 , "Matrix Ccont is not positive-semi-definite.")
  error(cols(Cid) <> rows(Cid), "Matrix Cid is not square.")
  error(cols(Ccont) <> rows(Ccont), "Matrix Ccont is not square.")
  error(dim(mid) <> dim (mcont), "Vector mcont and vector mid mismatch.")
  error(cols(Cid) <> dim (mid), "Matrix Cid and vector mid mismatch.")
  error(cols(Ccont) <> dim (mcont), "Matrix Ccont and vector mcont mismatch.")
  error((eps>=1)||(eps<0), "Contamination Radius not in [0,1).")
  error((DirNorm<>1)&&(DirNorm<>0)&&(DirNorm<>-1), "wrong Contamination Type.")

Ind=uniform(T)
Ind=Ind<eps
;Ind == 1 <=> contamination,    == 0  <=> no contamination

; calculating the sym.pos.semidef. root of Cid, Ccont for simulation
; if quantlet "symroot" is loaded, too
; one could instead write :
;
;Cidw=symroot(Cid)
;Ccontw=symroot(Ccont)
;
  Cidw=Cids.u*(sqrt(Cids.l).*Cids.v')
  Ccontw=Cconts.u*(sqrt(Cconts.l).*Cconts.v')

Xid=(Cidw*normal(T,rows(Cid))' + mid)'

switch
    case(DirNorm==0)
    Xcont=(Ccontw*normal(T,rows(Ccont))' + mcont)'
    break
    case(DirNorm==1)
    Xcont=(2*(uniform(T)>0.5)-1).*mcont'
    ; contamination is symmetrisized by randomly switching the sign
    break
    case(DirNorm==-1)
    Xcont=0*uniform(T)+mcont'
    break
endsw
  
X=(1-Ind).*Xid+Ind.*Xcont
endp
