proc(y,mu) = genmultlo(x,b)
; -----------------------------------------------------------------
; Library       glm
; -----------------------------------------------------------------
;  See_also     cumsum genglm glmmultlo
; -----------------------------------------------------------------
;   Macro       genmultlo
; -----------------------------------------------------------------
;   Description  genmultlo generates data according to a 
;                multinomial logit model with P( Y = j | Xa , Xi)
;                proportional to exp( Xa * ba + Xi * bi[j] ). Here, 
;                Xi denotes the part of the explanatory variables
;                which merely depends on the individuals, Xa covers
;                variables which may vary with the alternatives j.
;                Either part, Xa or Xi, can be omitted.
; -----------------------------------------------------------------
;   Link  ../tutorials/glmstart.html Tutorial: GLM in XploRe 
; ----------------------------------------------------------------------------
;   Usage        {y,mu} = genmultlo(x,b)
;   Input
;     Parameter   x  
;     Definition        n x r  matrix, design.
;     Parameter   b 
;     Definition        r x m matrix (m>1, first column equals zero)
;                       or r/m x 1 vector 
;                       or a list containing a p x m matrix (m>1, 
;                          first column equals zero) and a q x 1 
;                          vector and r=p+(q*m).
;   Output
;     Parameter   y
;     Definition        n x 1  vector, multinomial in 0 ... (m-1).
;     Parameter   mu
;     Definition        n x m  vector, response probabilities 
;                       P(y=j), j=1,...,m.
; -----------------------------------------------------------------
;   Example   library("glm")
;             n = 100
;             b = (0|0)~(1|2)~(-1|0)~(1|1)
;             x = normal(n,rows(b))
;             {y,mu} = genmultlo(x,b)
; -----------------------------------------------------------------
;   Result    y[i] are pseudo random variables with distribution 
;             P(y[i]=j) = exp(x[i,]*b[,j])/(1+sum(exp(x[i,]*b),2))
;             (= multinomial logit model).
; -----------------------------------------------------------------
;   Example   library("glm")
;             n = 100
;             b = 2
;             m = 4
;             id=diag(matrix(m))
;             x=kron(id,matrix(n/m))
;             {y,mu} = genmultlo(x,b)
; -----------------------------------------------------------------
;   Result    y[i] are pseudo random variables with distribution 
;             P(y[i]=j) = exp(x[i,j]*b)/(1+sum(exp(x[i,]*b),2))
;             (= conditional logit model).
; -----------------------------------------------------------------
;   Example   library("glm")
;             n = 100
;             b = list(2,(0|0)~(1|2)~(-1|0)~(1|1))
;             m = 4
;             id=diag(matrix(m))
;             x=kron(id,matrix(n/m))~normal(n,2)
;             {y,mu} = genmultlo(x,b)
; -----------------------------------------------------------------
;   Result    y[i] are pseudo random variables with distribution 
;             P(y[i]=j) = exp(x[i,j]*b{1} + x[i,5:6]*b{1}[,j])
;                         /(1+sum(exp(x[i,]*b{1}+exp(x[i,5:6]*b),2))
;             (= mixed multinomial/conditional logit model).
; -----------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/18
; -----------------------------------------------------------------
  n = rows(x)
  r = cols(x)
;
  switch
    case (exist(b)==9)               ; multinomial/conditional mixed
      bad=(cols(b{1})!=1)&&(cols(b{2})!=1)
      error(bad,"one parameter component is wrong!")
      if (cols(b{2})!=1)
        ba=b{1}
        bi=b{2}
        ifirst=1
      else
        bi=b{1}
        ba=b{2}
        ifirst=2
      endif
      bad=(cols(ba)!=1)||(cols(bi)==1)
      error(bad,"one parameter component is wrong!")
      p=rows(bi)
      m=cols(bi)
      q=rows(ba)
      bad=(cols(x)!=p+(q*m))
      error(bad,"design matrix has wrong number of columns!")
      if (ifirst==2)
        xi=x[,1:p]
        xa=x[,p+1:p+(q*m)]
      else
        xa=x[,1:q*m]
        xi=x[,1+q*m:p+(q*m)]
      endif
      id=diag(matrix(m))
;      id[1,1]=0
      ba=kron(ba,id)
      tmp=exp(xi*bi+xa*ba)
      mu=tmp./sum(tmp,2)
      break
    case ((exist(b)==1)&&(cols(b)==1)) ; conditional logit
      q=rows(b)
      m=cols(x)/q
      error(ceil(m)!=m,"design and parameter have incompatible dimensions")
      id=diag(matrix(m))
;      id[1,1]=0
      b=kron(b,id)
      tmp=exp(x*b)
      mu=tmp./sum(tmp,2)
      break
    case ((exist(b)==1)&&(cols(b)>1)) ; multinomial logit
      m = cols(b)
      error(r!=rows(b),"design and parameter have incompatible dimensions")
      tmp=exp(x*b)
      mu=tmp./sum(tmp,2)
      break
    default;
      error(0==0,"definition of parameter is wrong!")
  endsw
;
  u=uniform(n)
  p=cumsum(mu,2)
  y=sum(u>p,2)
endp




