proc (testt, val, df, pval)=haztest(data, index)   
; -----------------------------------------------------------------
; Library     hazreg
; -----------------------------------------------------------------
; See_also    hazdat, hazregll, hazbeta
; -----------------------------------------------------------------
; Macro       haztest
; -----------------------------------------------------------------
; Description calculates the value of the test statistic, the 
;             degree of freedom, and the P-value for the likelihood
;             ratio test, Wald's test and the score test for  Cox's 
;             Proportional Hazards model.
; -----------------------------------------------------------------
; Usage       {testt, val, df, pval} = haztest(data, index)
; Input       
; Parameter   data 
; Definition  n x (p+4) matrix, the first column is the sorted 
;             survival time t, followed by the sorted delta, 
;             inidcating if censoring has occured, labels l, a 
;             column containing the number of ties, and lastly, 
;             the sorted covariate matrix z.
; Parameter   index
; Definition  p x 1 vector, with ith element = 0 when beta[i] = 0
;             is in the null hypothesis, and 1, otherwise. 
; Output
; Parameter   testt 
; Definition  a table with values of the test statistics,
;             degrees of freedom, and p-values for the likelihood
;             ratio test, Wald's test and the score test.
; Parameter   val  
; Definition  3 x 1 vector, values of the test statistics.
; Parameter   df 
; Definition  scalar, degree of freedom.
; Parameter   pval 
; Definition  3 x 1 vector, P-values of the tests. 
; -----------------------------------------------------------------
; Example  library("hazreg") 
;            n = 20
;            p = 2
;            beta = 1|2                    ; regression parameter
;            z = 1 + uniform(n,p)          ; covariates
;            y = -log(1-uniform(n))        ; exponential survival
;            y = y./exp(z*beta)            ; covariate effects
;            c = 4*uniform(n)              ; uniform censoring
;            t = min(y~c,2)                ; censored time             
;            delta = (y<=c)                ; censoring indicator            
;            
;         {data,ties} = hazdat(t,delta, z) ; preparing data
;         index = 1|0                      ; testing whether the second
;                                          ; coefficient is zero
;         {testt, val, df, pval} = haztest(data, index)  
;         testt                            ; test results                              
; -----------------------------------------------------------------
; Result     The second parameter of beta is tested to be zero or 
;            not by likelihood ratio, Wald's and score tests. 
; -----------------------------------------------------------------
; Keywords   likelihood ratio test, Wald's test, score test,
;            Cox proportional hazards model
; -----------------------------------------------------------------
; Author     Lijian Yang, B. Grund 000307
; -----------------------------------------------------------------
  n=rows(data)

  p = cols(data) - 4
  
  q = p - sum(index)

  z = data[,5:p+4]
  z = z'

  z2 = paf(z,index)
  
  z2 = z2'
  z = z'

  data2 = data[,1:4]
  data2 = data2~z2
  
  {beta2hat,beta2k,c2k} = hazbeta(data2)  ;beta2 under H_0
  {betahat,betak,ck} = hazbeta(data)      ;beta  under H_a
  
{l2l,l2l1,l2l2} = hazregll(data2,beta2hat)

{ll,ll1,ll2} = hazregll(data,betahat)  

val1 = 2*(ll-l2l)               ;LR test statistic

; Compute Wald test statistic

ind = matrix(p,1) - index       ;index for beta1 subvector
   
beta1hat = paf(betahat,ind)     ;beta1 subvector under H_a
 
ll2 = inv(ll2)                  ;inverse of information matrix  

ll2 = paf(ll2,ind)

ll2 = paf(ll2',ind)             ;submatrix of inverse info. matrix

val2 = (beta1hat')*(inv(ll2))*beta1hat  ;Wald test statistic

; Compute score test statistic

beta0 = matrix(q,1)*0
beta0hat = beta0 | beta2hat

{ll,ll1,ll2} = hazregll(data,beta0hat) 

ll2 = inv(ll2)                  ;inverse of information matrix  

ll2 = paf(ll2,ind)

ll2 = paf(ll2',ind)             ;submatrix of inverse info. matrix

u1 = paf(ll1, ind)              ;subvector of first derivative

val3 = (u1')*(ll2)*u1

df = q

val = val1|val2|val3

pval = 1 - cdfc(val,q) 

hline ="-----------------------------------------------------------------"
hlined="- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -"

 text1 ="Cox Proportional Hazards Model"
 text2 ="Hypothesis: beta1=0 for a subvector of regression coefficients"

 vec1 = ("               ")+("   Test statistic   ")+("  DF   ")
 vec1 = vec1+("     P-value           ")
 
 mat1 = ("LR Test     ") + string("%16.5f", val1) + string("%10.0f", q)
 mat1 = mat1 + string("%16.5f",pval[1])

 mat2 = ("Wald Test   ") + string("%16.5f", val2) + string("%10.0f", q)
 mat2 = mat2 + string("%16.5f",pval[2]) 

 mat3 = ("Score Test  ") + string("%16.5f", val3) + string("%10.0f", q)
 mat3 = mat3 + string("%16.5f",pval[3])

 mat = mat1|mat2|mat3

 testt = hline|text1|""|text2
 testt = testt|""|hlined|vec1|hlined
 testt = testt|mat|hline|""

endp


