proc(t,p)=hhtest(vhat,y,yhat,h,c,m) 
;------------------------------------------------------------------------
; Library       metrics 
;------------------------------------------------------------------------
; See_also      hhmult glmest sker 
;------------------------------------------------------------------------
; Macro         hhtest 
;------------------------------------------------------------------------
; Description   hhtest calculates the H-H statistic to test the specifi-  
;               cation of the link function of a generalized linear 
;               model (such as the logit or probit model), assuming 
;               the index is correctly specified. 
; ----------------------------------------------------------------------- 
; Reference     Horowitz and Haerdle (1994), "Testing a parametric model 
;		against a semiparametric alternative", 
;		Econometric Theory,10, 821-848
; -----------------------------------------------------------------
; Usage         {t,p} = hhtest(vhat,y,yhat,h,{c,{m}})                  
; Input                                                                     
;   Parameter   vhat
;   Definition  n x 1  matrix with the projected index              
;   Parameter   y     
;   Definition  n x 1  matrix                                       
;   Parameter   yhat  
;   Definition  n x 1  matrix with the parametric estimate of E(Y|X)
;   Parameter   h     
;   Definition  scalar (positive) -- the bandwidth for kernel       
;               regression with Quartic kernel 
;   Parameter   c
;   Definition  scalar  0 =< c < 1 (optional)  -- proportion of the 
;               sample to be cut in each extreme. Default is 0.05.  
;   Parameter   m     
;   Definition  n x 1  or the scalar 1. m should be given only for  
;               binary responses. If the data is binomial m is the  
;               vector with the binomial coeficients. If the data is
;               bernouli, m=1. m is necessary to calculate the vari-
;               ance of y. If y is not binary the variance will be  
;               given by a nonparametric regression of  (y-fhat)^2  
;               on vhat.                                            
; Output                                                                     
;   Parameter   t     
;   Definition  scalar  -- the statistic value                      
;   Parameter   p     
;   Definition  scalar  -- the p-value of t                         
; ------------------------------------------------------------------------
;  Example                                                 
;            library("metrics") 
;            x = read("kyphosis.dat")                                        
;            y = x[,4]                                                 
;            x = x[,1:3]                                               
;            x = matrix(rows(x))~x                                     
;            h = 2                                                     
;            myfit=glmest("bilo",x,y)
;	     eta=x*myfit.b
;	     mu=1./(1+exp(-eta))	
;            {t,p} = hhtest(eta,y,mu,h,0.05,1)           
;            t~p
;-------------------------------------------------------------------------
;   Result    test statistic, t, and p-value, p.
; -----------------------------------------------------------------
; Author     Axel Werwatz,960326, based on code written by Isabel Proenca, 940724 
;-------------------------------------------------------------------------
 
error(cols(y)<>1, ": COLS of Y<>1")
error(cols(yhat)<>1, ": COLS of YHAT<>1")
error(cols(vhat)<>1, ": COLS of VHAT<>1")
x = vhat~y~yhat
n = rows(x)
x = matrix(n)~x
x = sort(x,2)
vhat = x[,2]
yhat = x[,4]
y    = x[,3]
cons = x[,1]
res1 =  y - yhat
s = h*n^(0.8/5)
br = (h/s)^2
; proportion to cut off in each side of the sample
        if (exist("c")==0)
                c = 0.05
        else
                error((c<0) || (c>=1), "wrong choice of c: set c such that 0 =< c < 1")        
        endif
numh = sker(vhat,h,"qua",y)
denh = sker(vhat,h,"qua",x[,1])
denh = denh - (n*h)^(-1).*0.9375     
numh = numh - (n*h)^(-1).*0.9375.*y  
denh = denh + (denh .= 0)
fhath = numh./denh
nums = sker(vhat,s,"qua",y)
dens = sker(vhat,s,"qua",x[,1])
dens = dens - (n*s)^(-1)*0.9375     
nums = nums - (n*s)^(-1)*0.9375.*y  
dens = dens + (dens .= 0)
fhats = nums./dens
fhat  = (fhath - br*fhats)./(1-br) ; remove bias from fhat
mhi   = vhat~fhat
res2  = fhat-yhat
; cut off the extreme values 
ind1 = ceil(c*n)+1  
ind2 = ceil((1-c)*n)  
res1  = res1[ind1:ind2]  
res2  = res2[ind1:ind2]  
; conditional variance of Y  
        if (exist("m")==1)
                error(cols(m)<>1 || (m<>1), "wrong choice of m")
                vy=m.*fhat.*(1-fhat)
        else            
                rs = (y - fhat).*(y - fhat)  
                numv = sker(vhat,h,"qua",rs)  
                denv = sker(vhat,h,"qua",x[,1])
                vy = numv./denv  
        endif
vy   = vy[ind1:ind2]  
;  
; H-H test statistic  
;  
t = sum(res1.*res2)  
;  
; variance of t  
;  
ck = 5/7  
phat = sker(vhat,h,"qua",x[,1])  
phat = phat[ind1:ind2]  
itg = sum(vy^2./phat)  
vt = 2*ck*n^(-1)*itg  
t = t/sqrt(vt)  
p = cdfn(t)    
endp
