proc (KP)=kpss(y)
; ----------------------------------------------------------------------
; Library 	  times
; ----------------------------------------------------------------------
;  See_also	  rvlm, lo, robwhittle, roblm, lobrob, gph, sclmtest, neweywest
;----------------------------------------------------------------------
;   Macro	  kpss
; ----------------------------------------------------------------------
;   Description   Calculation of the KPSS statistics for I(0) against long-memory
;                 alternatives. We consider the two tests, denoted by KPSS_mu and KPSS_t,
;                 and respectively based on a regression on a constant mu, and
;                 on a constant and a time trend t. As in the KPSS paper, we
;                 consider for the autocorrelation consistent variance 
;                 estimator the truncation lags denoted by L0, L4, and L12.
;                 The quantlet returns the value of the truncation lag, the 
;                 type of the test, i.e., const or trend, the estimated statistic, 
;                 and the critical value for a 95 percent confidence interval for I(0). 
;                 If the value of the test exceeds the critical value, a star symbol
;                 * is displayed after the test statistic.
;
; ----------------------------------------------------------------------
;   Usage         k = kpss(y)
;   Input 
;     Parameter   y 
;     Definition  vector
;     Output
;     Parameter   k
;     Definition  vector
; ----------------------------------------------------------------------
;   Example     ;Non-stationarity test on the series dmus58.dat. Since the evaluated
;               ;statistics are over the critical value for I(0), for all the
;               ;truncation orders, the star symbol is displayed after all the
;               ;computed statistics.
;               library("times")
;               x = read("dmus58.dat") ; 25477 obs: log FX rates
;               x = x[1:1000]
;               k = kpss(x) ; 
;               k
; ----------------------------------------------------------------------
;   Result
;                 Contents of k
;
;                [1,] "   Order   Test   Statistic  Crit. Value "
;                [2,] "_______________________________________ "
;                [3,] ""
;                [4,] " L0 =  0   const   77.7016 *   0.4630"
;                [5,] " L4 =  7   const    9.7786 *   0.4630"
;                [6,] " L12= 21   const    3.5996 *   0.4630"
;                [7,] " L0 =  0   trend   11.1167 *   0.1460"
;                [8,] " L4 =  7   trend    1.4222 *   0.1460"
;                [9,] " L12= 21   trend    0.5412 *   0.1460"
;
; ----------------------------------------------------------------------
;   Reference
;       Kwiatkowski,D., Phillips, P.C.B., Schmidt, P., and Y. Shin (1992):
;       Testing the Null Hypothesis of Stationarity Against the Alternative
;       of a Unit Root: How Sure Are We that Economic Series Have a Unit
;       Root, Journal of Econometrics, 54, 159-178.
; ----------------------------------------------------------------------
;  Keywords
;       long memory, tests, nonstationarity
; ----------------------------------------------------------------------
;   Author        Gilles Teyssiere, 980425. Revised 990120
; ----------------------------------------------------------------------
  ;;
  n = rows(y) 
  rat1 = (n/100)^(0.25)
  bandwidth = #(0,rint(4*rat1),rint(12*rat1))
  ;;
  temps = cumsum(matrix(n))
  ctemps = temps - mean(temps)
  cy = y - mean(y)
  betahat = sum(ctemps.*cy)/sum(ctemps^2)  
  constanthat = mean(y) - betahat*mean(temps)
  e1 = cy
  e2 = y - constanthat -betahat*temps
  S1 = sum(cumsum(e1)^2)
  S2 = sum(cumsum(e2)^2)
  ;;
  KP = matrix(6)
  sac1 = neweywest(e1,bandwidth)
	sac2 = neweywest(e2,bandwidth)
  KP[1] = S1/sac1[1]
  KP[2] = S1/sac1[2]
  KP[3] = S1/sac1[3]
  KP[4] = S2/sac2[1]
  KP[5] = S2/sac2[2]
  KP[6] = S2/sac2[3]
  KP = KP/(n^2)
	CI = matrix(6)
	i = 1
	while (i <= 3)
		if (KP[i] <= 0.463); critical value of the KPSS test
			CI[i] = 0
		else
			CI[i] = 42
		endif
		i = i+1;
	endo
  while (i <= 6)
		if (KP[i] <= 0.146); critical value of the KPSS test for trend I(0)
			CI[i] = 0
		else
			CI[i] = 42
		endif
		i = i+1;
	endo
	critvalue = #(0.463, 0.463, 0.463,0.146, 0.146,0.146)
  ;; 
	;; Display of the results
  line1 ="   Order   Test   Statistic  Crit. Value "
  line2 ="_________________________________________ "
  st0 = #(" L0 = "," L4 = "," L12= "," L0 = "," L4 = "," L12= ")
  st1 = string("%2.0f",bandwidth|bandwidth)
  st2 = #("   const","   const","   const","   trend","   trend","   trend")
  st3 = string("%10.4f",KP)
	st4 = string("%9.4f",critvalue)
  st5= char(matrix(6,2)*32)
	st6 = char(CI)
  st7 = st0+st1+st2+st3+st5+st6+st4
  KP = line1|line2|""|st7
endp
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; end of macro KPSS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
