proc (gamma, mu, sigma) = mlegp (x, k)
; -----------------------------------------------------------------
;   Library       finance
; -----------------------------------------------------------------
;   See_also      pickandsgp dpgp mlegp0 hillgp1 momentgp mlegpdiag
; -----------------------------------------------------------------
;   Macro         mlegp
; -----------------------------------------------------------------
;   Description   Maximum likelihood estimator for GP model
; -----------------------------------------------------------------
;   Usage         {gamma, mu, sigma} = mlegp(x, k)
;   Input
;     Parameter   x
;     Definition      vector
;     Parameter   k
;     Definition      integer, number of upper extremes
;   Output
;     Parameter   gamma
;     Definition      scalar, estimated shape parameter
;     Parameter   mu
;     Definition      scalar, estimated location parameter
;     Parameter   sigma
;     Definition      scalar, estimated scale parameter	    
; -----------------------------------------------------------------
;   Example   library("finance")
;             randomize(0)
;	            x=randx("gp",100,1)
;             m=mlegp(x,50)
;             m
; -----------------------------------------------------------------
;   Result    Contents of m.gamma
;             [1,]   1.3339 
;	            Contents of m.mu
;	            [1,]  0.15342 
;             Contents of m.sigma
;             [1,]  0.53523 
; -----------------------------------------------------------------
;   Author    Michael Thomas  990503
; -----------------------------------------------------------------
    n = rows (x)
    error (n < 8, "Data set must have at least 8 elements")
    if (k < 8)
        warning (k < 8, "MLE(GP) needs at least 8 exceedances")
        k = 8
    endif
    if (k > n)
        warning (k > n, "Number of exceedances is too large")
        k = n
    endif

    gamma = 0
    mu = 0
    sigma = 0
    if (getenv ("os")=="windows")
        r = dlcall ("_mlegp", x, rows (x), k, gamma, mu, sigma)
    else
        r = dlcall ("mlegp", x, rows (x), k, gamma, mu, sigma)
    endif
    warning (r != 0, "MLE(GP) failed")
endp
