proc (mu, sigma) = mlegp0 (x, k)
; -----------------------------------------------------------------
;   Library       finance
; -----------------------------------------------------------------
;   See_also      pickandsgp dpgp mlegp hillgp1 momentgp 
; -----------------------------------------------------------------
;   Macro         mlegp0
; -----------------------------------------------------------------
;   Description   Maximum likelihood estimator for exponential (GP0) model
; -----------------------------------------------------------------
;   Usage         {mu, sigma} = mlegp0 (x, k)
;   Input
;     Parameter   x
;     Definition      vector
;     Parameter   k
;     Definition      integer, number of upper extremes
;   Output
;     Parameter   mu
;     Definition      scalar, estimated location parameter
;     Parameter   sigma
;     Definition      scalar, estimated scale parameter	    
; -----------------------------------------------------------------
;   Example   library("finance")
;             randomize(0) 
;	            x=randx("gp0",100)
;             m=mlegp0(x,50)
;             m
; -----------------------------------------------------------------
;   Result    Contents of m.mu
;	            [1,] -0.15354
;             Contents of m.sigma
;             [1,]   1.0401
; -----------------------------------------------------------------
;   Author    Michael Thomas  990503
; -----------------------------------------------------------------
  n = rows (x)
  if (k < 2)
    warning (k < 2, "MLE(GP) needs at least 3 exceedances")
    k = 2
  endif
  if (k > n)
    warning (k > n, "Number of exceedances is too large")
    k = n
  endif
  x = sort (x)
  sigma = mean (x [n - k + 2:n]) - x [n - k + 1]
  mu = x [n - k + 1] - sigma * log (n / k)
endp
