proc(output)=panhaus(z,siga,sige,m,T)  
; ----------------------------------------------------------------------
; Library 	   metrics   
; ----------------------------------------------------------------------
; See_also	   panfix, panhaus, pandyn
; ----------------------------------------------------------------------
; Macro	       panhaus
; ----------------------------------------------------------------------
; Description  Hausman Test for a correlation between the 
;              regressors and the individual effects
; ----------------------------------------------------------------------
; Usage        output = panhaus(Z,siga,sige,m {,T} )
;
;   Input 
;     Parameter   Z  
;     Definition     Dataset ordered by Individuals (i) 
;                    1st column: Index of Individuals
;                    2nd column: Dependent variable
;                    3rd ... column: Explanatory variables (without const!)
;                    NT x k+2 Matrix 
;     Parameter   siga 
;     Definition     variance estimate of individual effects
;     Parameter   sige 
;     Definition     variance estimate of the remaining error
;     Parameter   m  
;     Definition     The first m explanatory variables are treated as corre-
;                    lated with the fixed effects, the remaining variables
;                    are assumed to be uncorrelated with the individual effects
;                    (= mixed specification)
;     Parameter   T  
;     Definition     Number of time periods for a BALANCED Dataset
;                    IMPORTANT: No column for the Index of Individuals!
;                    (Optional, only for a Dataset with the same T
;                    for all Individuals = BALANCED PANEL)     
;
;   Output
;     Parameter   output
;     Definition     String of output table
; ----------------------------------------------------------------------
;   Example       library("metrics")
;                 z=read("earnings")
;                 z1 =z[,1:2]~log(z[,3])~z[,4]~(z[,4]^2)
;                 {output,siga,sige}=panfix(z1,2)
;                 panhaus(z1,siga,sige,2) 
; ----------------------------------------------------------------------
;	Result
;	[ 1,] "====================================================="
;	[ 2,] "Hausman Specification Test for the null hypothesis:  "
;	[ 3,] " 2 Variables x(i,t) are uncorrelated with a(i)      "
;	[ 4,] "====================================================="
;	[ 5,] "  d = beta(between)-beta(within)    SE        t-value"
;	[ 6,] "====================================================="
;	[ 7,] "  d [ 1 ] =        -0.0705       0.0402        -1.754"
;	[ 8,] "  d [ 2 ] =         0.0001       0.0008         0.113"
;	[ 9,] "====================================================="
;	[10,] "P-Value (RANDOM against FIXED effects):        0.0000"
;	[11,] "====================================================="
; ----------------------------------------------------------------------
;   Author        Joerg Breitung, 990324
; ----------------------------------------------------------------------

  NT=rows(z)
  z=z~matrix(NT,1)  
  k=cols(z) 
 
  if (exist(T)) 
    N=NT/T
    error(N-floor(N)>0,"  The data set is not balanced (NT/N is not an integer)")
    error(T<2,         "  T must be at least 2")
    ind=(1:N)  
    ind=kron(ind,matrix(T,1))
    z=ind~z
    k=k+1
  else
    z=z[,1]~z[,3:k]
    k=k-1 
	endif	 

; Eingabeanalyse

    error(m>k-3,       "  m must be smaller than the number of variables")
    error(m<1  ,       "  m must be at least 1")
    error(siga<0,      "  siga must be positive")
    error(sige<0,      "  sige must be positive")

  sxx=0  
  sxy=0
  syy=0 
  sxx1=0
  sxy1=0
  Tm=0

  pos=1
  N=0
  while(pos < NT-1)
    ind=z[pos,1]
    yi=z[pos,2] 
    xi=z[pos,3:k]
    pos=pos+1
    while (z[pos,1]==ind)
      yi=yi|z[pos,2]
      xi=xi|z[pos,3:k]
      pos=pos+1
    endo

    Ti=rows(yi)
      error(Ti<2,"   T[i] must be at least 2")
    alpha=1-sqrt(sige/(sige+siga*Ti));
    mx=mean(xi)'
    my=mean(yi)
    xi=xi-alpha*mx' 
    yi=yi-alpha*my
    xi1=(1-alpha)*kron(matrix(Ti,1),mx[1:m]')   
    xi1=xi~xi1
    sxx=sxx+xi'*xi
    sxx1=sxx1+xi1'*xi1
    sxy=sxy+xi'*yi
    sxy1=sxy1+xi1'*yi
    syy=syy+yi'*yi
    N=N+1  
  endo
 
  b0=inv(sxx)*sxy 
  b1=inv(sxx1)*sxy1
  sse0=syy - b0'*sxy
  sse1=syy - b1'*sxy1
  sig0=sse0/(NT-rows(b0))
  sig1=sse1/(NT-rows(b1))
  SEb1=sig1*inv(sxx1) 
  SEb1=sqrt(xdiag(SEb1))	
  k=rows(b0) 
  F=(sse0-sse1)/sig1/m 
	FP=1-cdff(F,m,NT-k-m)

  betastr=string("  d [%2.0f ] =%15.4f", 1:m, b1[k+1:k+m])	
	betasestr=string("     %8.4f",SEb1[k+1:k+m])
	tvalues=b1./SEb1
	tvalstr=string("  %12.3f", tvalues[k+1:k+m])  
  FPstr  =string("     %8.4f",FP)

//TEXTOUTPUT	

  line0="Hausman Specification Test for the null hypothesis:  "
  line1=string("%2.0f Variables x(i,t) are uncorrelated with a(i)      ", m)
	line2="=====================================================" 
	line3="  d = beta(between)-beta(within)    SE        t-value"    
  line4="P-Value (RANDOM against FIXED effects): "+FPstr 
	parametertext=betastr+betasestr+tvalstr
	output=line2|line0|line1|line2|line3|line2|parametertext|line2|line4|line2

endp   

