proc (gamma, mu, sigma) = pickandsgp (x, k)
; -----------------------------------------------------------------
;   Library       finance
; -----------------------------------------------------------------
;   See_also      dpgp mlegp mlegp0 hillgp1 momentgp pickandsgpdiag
; -----------------------------------------------------------------
;   Macro         pickandsgp
; -----------------------------------------------------------------
;   Description   Pickands estimator for GP model.
; -----------------------------------------------------------------
;   Usage         {gamma,mu,sigma} = pickandsgp(x,k)
;   Input
;     Parameter   x
;     Definition      vector
;     Parameter   k
;     Definition      integer, number of upper extremes
;   Output
;     Parameter   gamma
;     Definition      scalar, estimated shape parameter
;     Parameter   mu
;     Definition      scalar, estimated location parameter
;     Parameter   sigma
;     Definition      scalar, estimated scale parameter	    
; -----------------------------------------------------------------
;   Example   library("finance")
;             randomize(0) 
;	            x=randx("gp",100,1)
;             m=pickandsgp(x,50)
;             m
; -----------------------------------------------------------------
;   Result    Contents of m.gamma
;             [1,]   1.6768 
;	            Contents of m.mu
;	            [1,]  0.34883 
;             Contents of m.sigma
;             [1,]  0.31658  
; -----------------------------------------------------------------
;   Author    Michael Thomas  990503
; -----------------------------------------------------------------
  n = rows (x)
  error (n < 4, "Data set must have at least 4 elements")
  if (k < 4)
    warning (k < 4, "Pickands estimator requires at least two exceedances")
    k = 4
  endif
  if (k > n)
    warning (k > n, "k is larger than the number of data points")
    k = n
  endif
  x = sort (x)
  m = floor (k / 4)
  gamma = log ((x [n - m + 1] - x [n - 2*m + 1]) / (x [n - 2*m + 1] - x [n - 4*m + 1])) / log (2)
  mu = x [n - k + 1]
  sigma = gpsigmaest (gamma, x [n - k + 1:n])
  r = fittail (gamma, mu, sigma, k, n)
  mu = r.mu
  sigma = r.sigma
endp