proc(hopt,ch)=regbwcrit(crit, x, h, K, d)
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      regbwsel regxbwsel regest
; -----------------------------------------------------------------
;   Macro        regbwcrit
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel regression,
;                bandwidth selection
; -----------------------------------------------------------------
;   Description  determines the optimal from a range of bandwidths 
;                by one using the resubstitution estimator with one
;                of the following penalty functions:
;                Shibata's penalty function (shi), 
;                Generalized Cross Validation (gcv),
;                Akaike's Information Criterion (aic),
;                Finite Prediction Error (fpe),
;                Rice's T function (rice).
;                The computation uses WARPing.
; -----------------------------------------------------------------
;   Reference    Haerdle (1990): Smoothing Techniques
;
;                Haerdle, Mueller, Sperlich, Werwatz (1999): Non- and
;                Semiparametric Modelling
; -----------------------------------------------------------------
;   Usage        {hopt, ch} = regbwcrit(crit, x {,h {,K} {,d} })
;   Input
;     Parameter  crit  
;     Definition     string, criterion for bandwidth selection:
;                    "shi", "gcv", "aic", "fpe", "rice".
;     Parameter  x  
;     Definition     n x 2 vector, the data.
;     Parameter  h  
;     Definition     m x 1 vector, vector of bandwidths.
;     Parameter  K  
;     Definition     string, kernel function on [-1,1] or Gaussian
;                    kernel "gau". If not given, "qua" is used.
;     Parameter  d  
;     Definition     scalar, discretization binwidth. d must be smaller
;                    than h. If not given, the minimum of min(h)/3 and 
;                    (max(x)-min(x))/500 is used.
;   Output
;     Parameter  hopt  
;     Definition     scalar, optimal bandwidth. 
;     Parameter  ch  
;     Definition     m x 2 vector, the criterion function for h values.
; -----------------------------------------------------------------
;   Example   library("smoother")
;             x=read("nicfoo")
;             h=grid(0.05,0.1,10)
;             {hopt,ch}=regbwcrit("gcv",x,h)
;             hopt
;             library("plot")
;             ch=setmask(ch,"line","blue")
;             plot(ch)
;             setgopt(plotdisplay,1,1,"title",string("hopt=%1.6g",hopt))
; -----------------------------------------------------------------
;   Result    hopt is the LSCV optimal bandwidth for these data.
;             The resulting curve for the LSCV criterion is plotted.
; -----------------------------------------------------------------
;   Author    Marlene Mueller, 990413
; ----------------------------------------------------------------- 
  error(cols(x)<>2, "regbwcrit: cols(x) <> 2")
;
  if (exist(K)==0)
    K="qua"
  endif
  if (exist(K)==1) ; no K but d
    d=K
    K="gau"
  endif
;
  if (exist(d)==0)
    d=(max(x[,1])-min(x[,1]))./500
    d=min(d~(min(h)./3),2)
  endif
;
  {xb,yb}=bindata(x[,1],d,0,x[,2])   ; bin data in x and sum of y's   
  y0=paf(yb,yb[,1]>0)
  w=y0[,1]
  y0=y0[,2]./y0[,1]                  ; "raw y data"
;
  n=rows(x)
  m=rows(h)
  dp0=_K(0).*d
  ch=matrix(m)-1
;
  i=0
  while (i<m)
    i=i+1
    if (K!="gau")
      wy=symweigh(0,d/h[i],h[i]/d,K)   ; create weights for kernel
    else
      wy=symweigh(0,d/h[i],5.*h[i]/d,K)
    endif    
    wx=aseq(0,rows(wy),1)
    {xc,yc,or}=conv(xb,yb,wx,wy)       ; smooth x's and y's   
;plot(xc~(yc[,2]./yc[,1]))
    yc=paf(yc,or!=0)
    mh=yc[,2]./yc[,1]
    dp=dp0./h[i]
    xi=dp./(yc[,1])
    xi=xi+ (max(xi)-min(xi)).*(1e-10)
;
    switch
      case (crit=="shi")
        xi= 1+2.*xi
        break;
      case (crit=="gcv")
        xi= 1./((1- xi)^2)
        break;
      case (crit=="aic")
        xi= exp(2.*xi)
        break;
      case (crit=="fpe")
        xi= (1+xi)./(1-xi)
        break;
      case (crit=="rice")
        xi= 1./(1-2.*xi)
        break;
      default;
        error(1,"denbwcrit: Criterion "+crit+" does not exist!")
        break;
    endsw
;
    res=(y0-mh)
    ch[i]=((res.*xi)'*res)./sum(yb[,1])
  endo
  ch=h~ch
  hopt = ch[minind(ch[,2]),1]
endp

