proc (mh)=regestp(x,h,K,d)   
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      regest lpregest
; -----------------------------------------------------------------
;   Macro        regestp
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel regression, 
;                Nadaraya-Watson estimator
; -----------------------------------------------------------------
;   Description  Nadaraya-Watson estimator for multivariate 
;                regression. The computation uses WARPing.
; -----------------------------------------------------------------
;   Notes        The WARPing enhances the speed of computation,
;                but may lead to computational errors, if the
;                bandwidth is small. For p<=2 WARPing is usually
;                faster than exact computation. For p>2, the
;                macro "regxestp" should be used instead.
; -----------------------------------------------------------------
;   Reference    Wand/Jones (1995): Kernel Smoothing
; -----------------------------------------------------------------
;   Usage        mh = regestp(x {,h {,K {,d}}})
;   Input
;     Parameter  x  
;     Definition   n x (p+1), the data. In the first p columns the
;                  independent, in the last column the dependent 
;                  variable.
;     Parameter  h 
;     Definition   scalar or p x 1 vector, bandwidth. If not 
;                  given, 20% of the volume of x[,1:p] is used.
;     Parameter  K  
;     Definition   string, kernel function on [-1,1]^p. If not given,
;                  the product Quartic kernel "qua" is used.
;     Parameter  d  
;     Definition   scalar, discretization binwidth. d[i] must be 
;                  smaller than h[i]. If not given, the minimum of h/3 
;                  and (max(x)-min(x))'/r, with r=100 for p=1, and 
;                  r=(1000^(1/p)) for p>1 is used.
;   Output
;     Parameter  mh  
;     Definition   m x (p+1) matrix, the first p columns constitute 
;                  a grid and the last column contains the regression
;                  estimate on that grid.
; -----------------------------------------------------------------
;   Example   library("smoother") 
;             library("plot")                                    
;             ;
;             x = 4.*pi.*(uniform(400,2)-0.5) 
;             m = sum(cos(x),2)
;             e = uniform(400)-0.5                          
;             x = x~(m+e)                             
;             ;
;             mh = regestp(x,2)
;             mh = setmask(mh, "surface","blue")
;             m  = setmask(x[,1:2]~m,"black","cross","small")
;             plot(mh,m)                                         
;             setgopt(plotdisplay,1,1,"title","ROTATE!")
; -----------------------------------------------------------------
;   Result    The Nadaraya-Watson regession estimate (blue) using   
;             Quartic kernel and bandwidth h=2 and the true 
;             regression function (thin black crosses) are pictured.
; -----------------------------------------------------------------
;   Author    Sigbert Klinke, 950425; Marlene Mueller, 990413
; -----------------------------------------------------------------
  error(cols(x)<2,"regestp: cols(x) < 2")   
  n=rows(x)
  p=cols(x)-1 
  ix=min(x[,1:p])
  mx=max(x[,1:p])
  rx=mx-ix
;
  if (exist(h)==0)
    h=((0.2)^(1/p)).*rx
  endif
  if (cols(h)>1)
    h=h'
  endif
  if (rows(h)==1)
    h = h.*matrix(p) 
  endif
;
  if (exist(K)==0)
    K="qua"
  endif
  if (exist(K)==1) ; no K but d
    d=K
    K="qua"
  endif
;
  if (exist(d)==0)
    r=100.*(p==1)+(1000^(1/p)).*(p>1)
    d=(max(x[,1:p])-min(x[,1:p]))'/r
    d=min(d~(h./3),2)
  endif
  if (cols(d)>1)
    d=d'
  endif
  if (rows(d)==1)
    d = d.*matrix(p,1) 
  endif
;
  error(sum(h .<=d)>0,"denestp: h smaller than d -- Please modify h or d!")
;
  w1 = matrix(p,1)
  wx = w1-1
  {xb,yb}=bindata(x[,1:p],d,wx,x[,p+1]) ; bin data in x and sum of y's   
  hd=h./d
  wy=symweigh(wx,d/h,hd,K)              ; create weights for quartic kernel
  wx=grid(wx,w1,hd)
  {xc,yc,or}=conv(xb,yb,wx,wy)          ; smooth x's and y's   
  mh=((xc+0.5).*d')~(yc[,2]./yc[,1]) 
;  mh=paf(mh,or!=0)
  mh=sort(mh,1:p)
endp


