proc (mh)=regxestp(x,h,K,v)   
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      regxest lregxestp regestp
; -----------------------------------------------------------------
;   Macro        regxestp
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel regression, 
;                Nadaraya-Watson estimator
; -----------------------------------------------------------------
;   Description  computes the Nadaraya-Watson estimator for 
;                multivariate regression. 
; -----------------------------------------------------------------
;   Notes        This function does an exact computation, i.e.
;                requires O(n^2) operations for estimating the 
;                regression function on all observations. For
;                p>2 this is usually faster than the WARPing method.
; -----------------------------------------------------------------
;   Reference    Haerdle (1990): Applied Nonparametric Regression
;
;                Wand/Jones (1995): Kernel Smoothing
;
;                Haerdle/Mueller (1999): Multivariate and
;                Semiparametric Kernel Regression
; -----------------------------------------------------------------
;   Usage        mh = regxestp(x {,h {,K} {,v} })
;   Input
;     Parameter  x  
;     Definition   n x (p+1), the data. In the first p columns 
;                  the independent variables, in the last column 
;                  the dependent variable.
;     Parameter  h 
;     Definition   scalar, p x 1 or 1 x p, bandwidth. If not 
;                  given, 20% of the range of x[,1:p] is used.
;     Parameter  K  
;     Definition   string, kernel function on [-1,1] or Gaussian
;                  kernel "gau". If not given, the Quartic kernel
;                  "qua" is used.
;     Parameter  v  
;     Definition   m x p, values of the independent variable on 
;                  which to compute the regression. If not given, 
;                  a grid of length 100 (p=1), length 30 (p=2)
;                  and length 8 (p=3) is used in case of p<4. When 
;                  p>=4 then v is set to x.
;   Output
;     Parameter  mh  
;     Definition   n x (p+1) or m x (p+1) matrix, the first p columns 
;                  contain the grid or the sorted  x[,1:p], the 
;                  second column contains the regression estimate  
;                  on the values of the first p columns.
; -----------------------------------------------------------------
;   Example   library("smoother") 
;             library("plot")
;             ;
;             x = 2.*pi.*(uniform(200,2)-0.5)  ; independent variable
;             m = sum(cos(x),2)                ; true function
;             e = uniform(200)-0.5             ; error term             
;             x = x~(m+e)                             
;             ;
;             mh = regxestp(x,2)               ; estimate function
;             mh = setmask(mh,"surface","blue")
;             plot(x,mh)                       ; surface plot                  
;             setgopt(plotdisplay,1,1,"title","ROTATE!")
; -----------------------------------------------------------------
;   Result    The Nadaraya-Watson regression estimate (blue) using   
;             Quartic kernel and bandwidth h=2 and the data are
;             pictured.
; -----------------------------------------------------------------
;   Author    Marlene Mueller, 990413
; -----------------------------------------------------------------
  n=rows(x)
  p=cols(x)-1
  ix=min(x[,1:p])
  mx=max(x[,1:p])
  rx=mx-ix
  one=matrix(p)
;
  if (exist(h)==0)
    h=((0.2)^(1/p)).*rx
  endif
  if (cols(h)>1)
    h=h'
  endif
  if (cols(h)==1)
    h=h.*matrix(p)
  endif
;
  if (exist(K)==0)
    K="qua"
  endif
  if (exist(K)==1) ; no K but v
    v=K
    K="qua"
  endif
;
  x=sort(x,1:p)
  y=x[,p+1]
  x=x[,1:p]
;
  if (exist(v)==0)
    switch
      case (p==1)
        v=grid(ix',rx'/99,100*one)
        break
      case (p==2)
        v=sort(grid(ix',rx'/29,30*one),1:p)
        break
      case (p==3)
        v=sort(grid(ix',rx'/7,8*one),1:p)
        break
      default
        v=x
        break
    endsw  
  endif
;
  if (K!="gau")
    tmp=sker(x,h',K,matrix(n)~y,v)
  else
    tmp=sker(x,5.*h',"ngau",matrix(n)~y,v)
  endif
  mh=v~ (tmp[,2]./tmp[,1])
endp


