proc(d)=roblm(y,arg1,arg2)
; ----------------------------------------------------------------------
; Library 	  times
; ----------------------------------------------------------------------
;  See_also	  robwhittle, lobrob, gph, lo, kpss, rvlm, sclmtest, pgram,
; ----------------------------------------------------------------------
;   Macro	  roblm
; ----------------------------------------------------------------------
;   Description   Semiparametric average periodogram estimator of the 
;                 degree of long memory of a time series.
;                 The first argument of the macro is the series, the second
;                 optional argument is a strictly positive constant q, which is
;                 also strictly less than one. 
;                 The third optional argument is the bandwidth vector m. 
;                 By default q is set to 0.5 and the bandwidth vector is equal to  
;                 m = n/4, n/8, n/16. If q and m contain several elements, 
;                 the estimator is evaluated for all the combinations of q and m.
;                 The quantlet returns in the first column the estimated degree of
;                 long-memory, in the second column the number of frequencies considered,
;                 in the third column the value of q. 
; ----------------------------------------------------------------------
;   Usage         d = roblm(x{,q{,bdvec}})
;   Input 
;     Parameter   x 
;     Definition  vector
;     Parameter   q
;     Definition  vector
;     Parameter   bdvec
;     Definition  vector
;     Output
;     Parameter   d
;     Definition  vector
; ----------------------------------------------------------------------
;   Example     ;nonparametric estimation of degree of long-memory in volatility
;               library("times")
;               x = read("dmus58.dat")
;               x=x[1:2000]
;               y = abs(tdiff(x)) 
;               d = roblm(y)
;               d
; ----------------------------------------------------------------------
;   Result
;                 Contents of d
;  
;                [1,] "     d     Bandwidth      q   " 
;                [2,] "_______________________________" 
;                [3,] "" 
;                [4,] "  0.0927      500        0.5" 
;                [5,] "  0.1019      250        0.5" 
;                [6,] "  0.1199      125        0.5" 
; ----------------------------------------------------------------------
; Example
;                library("times")
;                x = read("dmus58.dat")
;                x = x[1:2000]
;                y = abs(tdiff(x[,1])) 
;                q = #(0.5,0.7)
;                d = roblm(y,q)
;                d
; ----------------------------------------------------------------------
; Result
;                 Contents of d
;
;                [1,] "     d     Bandwidth     q   " 
;                [2,] "______________________________" 
;                [3,] "" 
;                [4,] "  0.0927      500       0.5" 
;                [5,] "  0.0750      500       0.7" 
;                [6,] "  0.1019      250       0.5" 
;                [7,] "  0.0810      250       0.7" 
;                [8,] "  0.1199      125       0.5" 
;                [9,] "  0.1230      125       0.7" 
; ----------------------------------------------------------------------
; Example
;                library("times")
;                x = read("dmus58.dat")
;                x = x[1:2000]
;                y = abs(tdiff(x[,1])) 
;                q = #(0.5,0.7)
;                m = #(250,500,750)
;                d = roblm(y,q,m)
;                d
; ----------------------------------------------------------------------
; Result
;                 Contents of d
;                [1,] "     d     Bandwidth     q   " 
;                [2,] "______________________________" 
;                [3,] "" 
;                [4,] "  0.1019      250       0.5" 
;                [5,] "  0.0810      250       0.7" 
;                [6,] "  0.0927      500       0.5" 
;                [7,] "  0.0750      500       0.7" 
;                [8,] "  0.0700      750       0.5" 
;                [9,] "  0.0876      750       0.7" 
; ----------------------------------------------------------------------
; Example
;                library("times")
;                x = read("dmus58.dat")
;                x = x[1:2000]
;                y = abs(tdiff(x[,1])) 
;                m = #(500,250,125)
;                d = roblm(y,m)
;                d
; ----------------------------------------------------------------------
; Result
;                 Contents of d
;                [1,] "     d     Bandwidth     q   " 
;                [2,] "_______________________________" 
;                [3,] "" 
;                [4,] "  0.0927      500       0.5" 
;                [5,] "  0.1019      250       0.5" 
;                [6,] "  0.1199      125       0.5" 
;
; ----------------------------------------------------------------------
;   Reference
;     Robinson, P.M. (1994): Semiparametric Analysis of Long-Memory 
;       Time Series, The Annals of Statistics, 22, 515-539.
; ----------------------------------------------------------------------
;   Keywords
;      long memory
; ----------------------------------------------------------------------
;   Author        Gilles Teyssiere, 981106
; ----------------------------------------------------------------------  
  ;;  default parameters
  n = rows(y)
  range = #(rint(n/4), rint (n/8), rint (n/16)); default bandwidths
  max1  = 3
  q     = #(0.5)
  max2  = 1 
  ;;tests the presence of the optional vectors of parameters
  targ1 = exist(arg1)
  targ2 = exist(arg2)
  if (targ1 != 0)
    if (targ1 == 1)     
      q  = arg1
      max2 = rows(arg1)
			if (min(q) > 1)
				q = #(0.5)
        range = #(rint(n/4), rint (n/8), rint (n/16)); default bandwidths			
      endif
      if (targ2 == 1)
        range = rint(arg2)
        max1 = rows(arg2)
      endif
    endif
  endif  
  ;;
  ;; This part of the code is from the macro pgram
  ;; written by Rong Chen and Christian Hafner
  y = y-mean(y)
  p = log(n) / log(2)
  p = rint(exp(log(2)*ceil(p))) ; next power of 2 
  if (n != p)
    y = y|(matrix(p-n)-1)       ; append series 
  endif
  y = y~(matrix(p).*0)
  ftr = fft(y)                  ; fourier transform
  I = (sum((ftr.*ftr)')/p)'     ; periodogram
  ind=2:(floor(n/2))
  I = I[ind]
  lambda = ind*2*pi/n
  ;;
  ;;   
  i  = 1
  ii = 1
  davper = matrix(max1*max2,3)
  while (i <= max1) 
    error(range[i] > n/2,"Bandwidth greater than the Nyquist frequency") 
    lm  = range[i]
		Flm = sum(I[1:lm])
		j = 1
		while (j <= max2)
      error(q[j] >= 1,"q should be strictly less than 1 and strictly positive")  
      error(q[j] <= 0,"q should be strictly positive and strictly less than 1")
      lamax = rint(q[j]*lm)
			Fqlm  = sum(I[1:lamax])		   	   	
      davper[ii,1] = 0.5-(log(Fqlm/Flm)/(2*log(q[j]))) 
			davper[ii,2] = lm
			davper[ii,3] = q[j]
			j = j+1
      ii = ii+1 
    endo
    i = i+1  
  endo
  ;;
	;; Display the results  
  line1 ="     d     Bandwidth     q   "
  line2 ="______________________________"
  st1 = string("%8.4f",davper[,1])
  st2 = string("%9.0f",davper[,2])
  st3 = string("%10.1f",davper[,3])
  st4 = st1+st2+st3
  davper = line1|line2|""|st4  
  d = davper
endp
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; end of macro roblm
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


