proc(a,b)=select(x,y,id,h)
;------------------------------------------------------------------------
; Library	metrics
;------------------------------------------------------------------------
; See_also	heckman   powell andrews  sssm  gennorm	
;------------------------------------------------------------------------
; Macro		select
;------------------------------------------------------------------------
; Description   select calculates semiparametric estimators of the 
;		intercept and slope coefficients in the "outcome" or "level" equation 
;		of a self-selection model. select is therefore the second stage
;	        of the two-stage procedure used to estimate these models. 
;		select does not estimate the coefficients of the "selection" or "decision" 
;		equation but requires that the estimated first-step "index" 
;		be given as an input. The procedure to estimate the slope coefficients 
;		is desribed in Powell (1987). The procedure to estimate the intercept 
;		coefficient is desribed in Andrews and Schafgans (1994). 
;		select combines the powell and andrews macros of the metrics library.
; ----------------------------------------------------------------------- 
; Usage		{a,b} =  select(x,y,id,h)                 
;------------------------------------------------------------------------
; Input                                                                     
; Parameter	x  
; Definition		n x M  regressor matrix. WARNING: x may not contain a vector of ones ! 
; Parameter	id  
; Definition		n x 1  vector containing the estimated index of the first-step selection equation.
; Parameter	y
; Definition		n x 1  matrix containing n observ. of the dependent variable                                      
; Parameter	h     
; Definition	        2 x 1  vector of bandwidth. the first element of h is the bandwidth 
;			       used for estimating the intercept coefficient while the second element of h 
;			       is the bandwidth used for estimating the slope coefficients.       
; Output                                                                     
; Parameter	a     
; Definition		scalar 	estimated intercept coefficient                 
; Parameter	b     
; Definition		M x 1 	vector of estimated slope coefficients                 
; ------------------------------------------------------------------------
;  Example 	
;		library("metrics")
;		randomize(66666)
; 		n	= 200				; sample size
;		ss1	= #(1,0.9)~#(0.9,1)		; covariance matrix of error terms
;		g	= #(1)				; true coefficient of decision equation
;		b	= #(-9, 1)			; true intercept and slope of outcome equation
;		u	= gennorm(n, #(0,0), ss1)	; generate realizations of joint distribution of error terms
;		ss2     = #(1,0.4)~#(0.4,1)		; covariance matrix of regressors
;		xz      = gennorm(n, #(0,0), ss2)       ; generate realizations of joint distribution of regressors
;		z       = xz[,2]			; regressor of decision equation
;		q       = (z*g+u[,1].>=0)		; generate binary dependent variable of decision equation
;		hd	= 0.1*(max(z) - min(z))		; bandwidth for dwade procedure
;		d	= dwade(z,q,hd)*(2*sqrt(3)*pi)	; dwade estimate * scaling factor
;		id	= z*d				; estimated first-step index
;		h       = (quantile(id, 0.7))|(0.2*(max(id) - min(id)))	;  bandwidth for select procedure
;		x	= matrix(n)~xz[,1]		; regressors for outcome equation
;		y       = x*b+u[,2]			; dependent variable for outcome equation
;		zz	= paf(y~x~id, q)		; impose censored sampling
;		y	= zz[,1]
;		x	= zz[,3:(cols(zz)-1)]
;		id	= zz[,cols(zz)]
;		{a,b}	= select(x,y,id,h)
;		d~a~b	; first-step estimate ~  intercept estimate ~ slope estimate				
; -----------------------------------------------------------------
;   Result    two-step estimates of a semiparametric
;             sample selection model according to Ahn and Powel (1993) and
;	      Andrews and Schafgans (1994)
;-------------------------------------------------------------------------
; Author	akw,961106 
; ----------------------------------------------------------------------------
		z=sort(id~y~x,1)
		n=rows(z)
		id=z[,1]
		z=z[,2:cols(z)]
		one=matrix(rows(id))
		num=sker(id, h[2,],"qua",z~one)
		num=num-15/16.*(z~one)./(n*h[2,])
		loo=num[,1:(cols(num)-1)]./(num[,cols(num)]+1e-10)
		yloo=z[,1]-loo[,1]
		xloo=z[,2:cols(z)]-loo[,2:cols(loo)]
		xins=z[,2:cols(z)].*(n*h[2,].*(num[,cols(num)]+1e-10))
		b=inv(xins'*xloo)*(xins'*yloo)
	u=id.-h[1,]			; (-index) - smoothing parameter
	l1=u.>=0		; =1 if (index) > smoothing parameter
	l2=u.>=3		;=1 if (index) > 3 ("3" is "b" in the paper)
	l3=matrix(rows(y))	;vector of "1"s
	l4=1-l2			;=1 if (index) < 3
        s=1-exp(-(u.*l1)./(3-(u.*l1)))  ;weight function s=1-exp(-u/(b-u)),
	s=s.*l4			; sets s=0 if (-index) > 3
	s=s+l2			; sets s=1 if (-index) > 3
	l5=s.<>0		; =1 if s > 0
	y=z[,1]
	x=z[,2:cols(z)]
	xbeta=x*b		;slopes * x for stayer`s earnings equation
	xx=(y-xbeta).*s		; numerator
	a=sum(xx)/sum(s)	;sum(xx)/sum(s) is alpha estimator
	endp
