proc (datt, risk) = stein(dat, var)
; -----------------------------------------------------------------------
; Library      wavelet
; -----------------------------------------------------------------------
; See_also     fwt invfwt sure
; -----------------------------------------------------------------------
; Link         ../tutorials/wavestart.html Wavelet tutorial
; -----------------------------------------------------------------------
; Macro        stein
; -----------------------------------------------------------------------
; Description  Stein computes the optimal threshold for a vector of data plus 
;              noise so that the mean squared error is minimized. Stein uses 
;              Stein's unbiased risk estimator for the risk. The quantlet
;              sure uses stein to threshold the father and mother wavelet
;              coefficients.   
; -----------------------------------------------------------------------
; Usage        {datt, risk} = stein (dat, var)
; Input
;   Parameter  dat 
;   Definition p vector of input data 
;   Parameter  var 
;   Definition scalar, variance   
; Output
;   Parameter  datt
;   Definition p vector, thresholded estimate
;   Parameter  risk
;   Definition scalar, minimal risk   
; -----------------------------------------------------------------------
; Example      ; loads the library wavelet
;              library ("wavelet")
;              var = 1
;              n = 16
;              ; computes a noisy function
;              x = 1:n
;              dat = var * normal (16)
;              ; thresholds dat
;              {datt, risk} = stein (dat, var)
;              d = createdisplay(1,1)
;              tdat = x~datt
;              setmaskl(tdat, (1:rows(tdat))', 4, 1, 3)
;              show(d, 1, 1, tdat,x~dat)
; -----------------------------------------------------------------------
; Result       denoised estimate and minimal risk
; -----------------------------------------------------------------------
; Author       Yurii Golubev, 970921, Sperlich 980105
; -----------------------------------------------------------------------
   
  t  = 1 : rows(dat)
  sqdat  = sort (dat^2)
  r = cumsum (sqdat) .- 2 .* t .* var .+ (rows(dat) .- t) .* sqdat
  k  = minind(r)
  sr = r[k]
  if(sr < 0.0)
    thres  = sqrt (sqdat[k])
    risk = sr + rows (dat) * var
    datt = (dat - thres) .* (dat .> thres) + (dat + thres) .* (dat .< -thres) 
  else 
     risk = rows (dat) * var
     datt = dat
  endif
endp
