; -----------------------------------------------------------------
; Library        cafpe
; -----------------------------------------------------------------
;  See_also      tp/cafpe/cafpefull, tp/cafpe/cafpedefault, tp/cafpe/cafpeload
; -----------------------------------------------------------------
;   Macro        cafpe
; -----------------------------------------------------------------
;   Keywords     nonparametric time series analysis, lag selection,
;                Final Prediction Error, CAFPE, local linear regression
; -----------------------------------------------------------------
;   Description Quantlet to conduct lag selection for the conditional
;               mean function in nonlinear autoregressive
;               models. It also allows for prior data transformations.
;               It uses local linear estimation for the
;               estimation of a corrected Asymptotic Final Prediction Error 
;               (CAFPE).
;               This quantlet does not allow to change advanced
;               parameter settings nor to select lags for the conditional
;               volatility function. For doing this, use the quantlet cafpefull.
;
; -----------------------------------------------------------------
;   Reference    Tschernig and Yang (2000), Nonparametric
;                Lag Selection for Time Series,
;                Journal of Time Series Analysis,
;                forthcoming  (http://www.blackwellpublishers.co.uk)
; -----------------------------------------------------------------
;   Usage        { crmin,crpro } = cafpe(y,truedat,xdataln,xdatadif,xdatastand,lagmax,searchmethod,dmax)
;   Input
;	 Parameter        ATTENTION:
;      Definition   this quantlet requires to open locling.dll, density.dll (NT)
;               or locling.so, denc.so (UNIX). This can be done with the quantlet cafpeload or 
;               directly with
;               garb = dlopen ("<path>\locling.dll") on NT,
;               garb = dlopen ("<path>\density.dll") on NT,
;               garb = dlopen ("<path>/locling.so") on UNIX,
;               garb = dlopen ("<path>/denc.so") on UNIX.
;     Parameter   y
;     Definition     n x 1 matrix, the observed time series, set to zero if truedat is used
;     Parameter   truedat
;     Definition      character variable that contains path and name
;                    of ascii data file if y=0
;     Parameter   xdataln
;     Definition  character variable: "yes": take natural logs, "no": don't
;     Parameter   xdatadif
;     Definition  character variable: "yes": take first differences of data, "no": don't
;     Parameter   xdatastand
;     Definition  character variable: "yes": standardize data, "no": don't
;     Parameter   lagmax
;     Definition  scalar variable with largest lag to be considered
;     Parameter   searchmethod
;     Definition  character variable: "full": consider all possible lag combinations,
;                     "directed": do directed search (recommended if lagmax > 10)
;     Parameter   dmax
;     Definition  scalar variable with maximum number of possible lags
;   Output
;     Parameter   crmin
;     Definition      vector that stores for all considered lag combinations
;                     in the first dmax columns the selected lag vector,
;                     in the dmax+1 column the estimated CAFPE,
;                     in the dmax+2 column the estimated true prediction error A,
;                     in the dmax+3 column the bias corrected estimate of the true prediction error A
;     Parameter   crpro
;     Definition      matrix that stores for each number of lags (0,1,...,dmax):
;                     in the first dmax colunms the selected lag vector,
;                     in the dmax+1 column the estimated asymptotically
;                      optimal bandwidth for estimating the true prediction error and CAFPE,
;                     in the dmax+2 column the bandwidth for estimating
;                      the constant B which is used for computing CAFPE and the plug-in bandwidth,
;                     in the dmax+3 column the used bandwidth for estimating
;                      the constant C which is used for computing the plug-in bandwidth,
;                     in the dmax+4 column the estimated CAFPE,
;                     in the dmax+5 column the estimated true prediction error A,
;                     in the dmax+6 column the bias corrected estimated true prediction error A
;; -----------------------------------------------------------------
;   Example    pathcafpe 	= "tp/cafpe/" 		; path for CAFPE quantlets
;		   library("xplore")
;		   library("times")
;		   func(pathcafpe + "cafpeload"); load required XploRe files of CAFPE
;	         cafpeload(pathcafpe)    	
;              n              = 50
;
;              truedat      = "";
;              xdataln      = "no";
;              xdatadif     = "no";
;              xdatastand   = "no";
;              lagmax       = 4
;              searchmethod = "full"
;              dmax         = 4
;
;              ; generate data
;              randomize(0)     ; sets seed value for random number generator
;              y=genexpar(1,0.1,0.3|0.6, 2.2|-0.8,normal(n))  ; generate exponential AR(2) process
;
;              ; plot generated data
;              title        = "Realisation of exponential AR(2) process"
;              xlabel       = "Time"
;              ylabel       = "Values"
;              plot(1:n~y)
;
;              ; conduct lag selection
;              { crmin,crpro } = cafpe(y,truedat,xdataln,xdatadif,xdatastand,lagmax,searchmethod,dmax)
;              "selected lag vector, estimated CAFPE "
;              crmin[,1:dmax+1]
;              "number of lags, chosen lag vector, estimated CAFPE, plug-in bandwidth"
;              (0:dmax)~crpro[,1:dmax|(dmax+4)|(dmax+1)]
; -----------------------------------------------------------------
;   Result    the selected lags according to the CAFPE critierion
;             and the estimated bandwidths
;             as described by Tschernig and Yang (2000)
; -----------------------------------------------------------------
;   Author    Tschernig  000420
; -----------------------------------------------------------------

proc (crmin,crpro) = cafpe(y,truedat,xdataln,xdatadif,xdatastand,lagmax,searchmethod,dmax)

; get default settings of advanced parameters
{ xresid,trueres,volat,startval,robden,selcrit,perA,perB,noutputf,outpath }=cafpedefault()

; call full routine
{ crmin,crpro,crstore,crstoreadd,hstore,hstoretest} = cafpefull(y,truedat,xresid,trueres,xdataln,xdatadif,xdatastand,lagmax,volat,searchmethod,dmax,selcrit,robden,perA,perB,startval,noutputf,outpath)


endp
