; -----------------------------------------------------------------
; Library        cafpe
; -----------------------------------------------------------------
;  See_also     tp/cafpe/fpenps 
; -----------------------------------------------------------------
;   Macro       fpenpsl
; -----------------------------------------------------------------
;   Keywords     nonparametric time series analysis, lag selection,
;                Final Prediction Error, CAFPE, local linear regression
; -----------------------------------------------------------------
;   Description Quantlet to compute lag selection criteria for
;               nonlinear autoregressive models for a given vector of lags.
;               It allows to compute two criteria based on
;               local linear estimation of the Asymptotic Final Prediction
;               Error: AFPE and CAFPE.
;    			If a scalar bandwidth is given, it is used as hA in the computation
;			of AFPE and CAFPE. 
;			If a vector bandwidth is given, only the residuals are computed
;			and zeros returned for the criteria.
; -----------------------------------------------------------------
;   Usage  { critoone,Ahat,sig2wnhat,hA,hB,hC,critgrid,xtj,yorig,resid } = fpenpsl(xraw,xresid,lags,lagmax,hAgiven,volat,startval,robden,estimator,kernel,selcrit,perA,perB,Ksqint)
;   Input
;	 Parameter          ATTENTION:
;      Definition  this quantlet requires to open locling.dll, density.dll (NT)
;               or locling.so, denc.so (UNIX). This can be done with the quantlet cafpeload or 
;               directly with
;               garb = dlopen ("<path>\locling.dll") on NT,
;               garb = dlopen ("<path>\density.dll") on NT,
;               garb = dlopen ("<path>/locling.so") on UNIX,
;               garb = dlopen ("<path>/denc.so") on UNIX.

;       Parameter   xraw
;       Definition  (n x 1) vector of univariate time series
;       Parameter   xresid
;       Definition  (nr x 1) vector of residuals for selecting lags
;                   of conditional volatility function; if not needed
;                   set xresid = 0
;       Parameter   lags
;       Definition  (d x 1) vector of lags
;       Parameter   lagmax
;       Definition  scalar, largest lag to be considered
;	Parameter   hAgiven
;	Definition  scalar bandwidth for which if set to zero a scalar bandwidth is computed using hoptest
;			or (d x 1) vector of bandwidth for which only the residuals are computed but no
;			critiera.
;       Parameter   volat
;       Definition  "no": lag selection for conditional mean function;
;                   "resid": lag selection for conditional volatility function,
;                   the residuals of fitting a conditional mean
;                   function have to be contained in xresid
;       Parameter   startval
;       Definition  character variable, to control treatment of starting values
;                   "different": for each lag vector as few starting
;                                values are used as necessary;
;                   "same": for each lag vector the same starting value is used which
;                           is determined by the largest lag used in the lag selection
;                           quantlet xorigxe
;       Parameter   robden
;       Definition  character variable for switching on/off robustification
;                               of density estimation a la Tjostheim & Auestad (1994),
;                               see also Section 5 in TY
;                               "yes": on; "no": off
;       Parameter   estimator
;       Definition  character variable for selection nonparametric estimator
;                   "loclin": local linear estimator; other estimators not
;                   implemented
;				Parameter   kernel
;       Definition  character variable, kernel used; "gaussian": Gaussian kernel
;       Parameter   selcrit
;       Definition  character variable to select lag selection critierion:
;                   "lqafpe":  estimating the asymptotic Final Prediction Error (AFPE) using
;                              local linear estimation and a  plug-in bandwidth based on partial
;                              local quadratic estimator
;                   "lqcafpe": estimating the corrected asymptotic Final Prediction Error (CAFPE) using
;                              local linear estimation and a  plug-in bandwidth based on partial
;                              local quadratic estimator
;       Parameter   perA
;       Definition  scalar, parameter used for screening off 0 <= perA <= 1 percent of the observations
;                   with the lowest density for estimating A, see  eq. (3.1) and Section 5 in TY
;       Parameter   perB
;       Definition  scalar, parameter like perA but for screening off perB observations
;                   with lowest density for estimating B
;       Parameter   Ksqint
;       Definition  scalar, constant of kernel ||K||_2^2 = integral of K(u)^2 du
;   Output
;       Parameter   critone
;       Definition  scalar, estimated criterion value (requires scalar bandwidth)
;       Parameter   Ahat
;       Definition  scalar, computed Ahat of TY, eq. (3.1) 
;       Parameter   sig2wnhat
;       Definition  scalar, estimated variance of white noise based on local linear estimation,
;                   see TY, eq. (3.4) (requires scalar bandwidth) 
;       Parameter   hA
;       Definition  scalar, estimated asymptotically optimal bandwidth for estimating A and (C)AFE
;       Parameter   hB
;       Definition  scalar, bandwidth for estimating B which is used  for plug-in bandwidth hA and
;                     (C)AFPE
;       Parameter   hC
;       Definition  scalar, bandwidth for estimating C which is used for plug-in bandwidth hA
;       Parameter   critgrid
;       Definition  vector, estimated criteria AFPE and CAFPE for a given lag vector (requires scalar bandwidth)
;	Parameter   xtj
;	Definition  (nr x d) matrix of lagged variables
;	Parameter   yorig
;	Definition  (nr x 1) vector of dependent observations
;       Parameter   resid
;       Definition  (nr x 1) vector of residuals which are estimated for the given lag vector

; -----------------------------------------------------------------
;   Reference Tjostheim, D. and Auestad, B. (1994) Nonparametric
;                           identification of nonlinear time series: selecting significant lags.
;                           Journal of the American Statistical Association 89, 1410-1419.
;
;                           Tschernig, R. and Yang, L. (TY) (2000), Nonparametric
;             Lag Selection for Time Series,
;             Journal of Time Series Analysis,
;             forthcoming  (http://www.blackwellpublishers.co.uk)
; -----------------------------------------------------------------
;   Author    Tschernig  000420
; -----------------------------------------------------------------
proc (critoone,Ahat,sig2wnhat,hA,hB,hC,critgrid,xtj,yorig,resid) = fpenpsl(xraw,xresid,lags,lagmax,hAgiven,volat,startval,robden,estimator,kernel,selcrit,perA,perB,Ksqint);

	d           = rows(lags)            ; number of lags of current specification
  selmet      = substr(selcrit,1,3)   ; extracts bandwidth and function
                                      ; stimation method from selcrit
                                      ; (ar,nw,lg,lq)
  sigy2      	= (var(xraw));

; ---------------------- prepare regressor matrices -----------------------

  { xsj,yorig }   = xorigex(xraw,xresid,lags,volat)

  if (startval == "same");
        { xsj,yorig } = xorigst(xsj,yorig,xraw,lagmax);
                                          /* drop lagmax starting values */
  endif;

  n           = rows(yorig);
  ntotal     	= rows(xraw);
	lagsso     	= sort(lags,1);

	xtj       	= xsj[1:rows(yorig),];

; -------------------- robustification of dens. est.---------------

  if (robden == "no")     ; to choose robustification of density estimation
        xsj   	= xtj       ; a la Auestad
  else
        xsj   	= xsj;
	endif;


; -------- compute plug-in bandwidth for computation of AFPE, CAFPE ----

	{ Bhat,Bhatr,hB,Chat,Chatr,hC,hA } = hoptest(xsj,yorig,xtj,estimator,kernel,ntotal,sigy2,perB,robden);

	if (prod(hAgiven)!=0)	    ; instead of plug-in bandwidth for hA a user specified bandwidth is applied
			hA	= hAgiven
	endif

; ----------------- compute selection criteria -------------------------

  { afpe,cafpe,sig2wnhat,Ahat,resid } = fpefsv(xtj,yorig,hA,xtj,kernel,estimator,Bhat,Chat,perA,Ksqint,ntotal);

  switch
  case (selcrit == "lqafpe");
      crito      = afpe|cafpe;
      break
  case (selcrit == "lqcafpe");
      crito      = cafpe|afpe;
      break
  default
      "check selected criterion";
      break
  endsw

  critgrid       = crito';
  hgrid          = hA;


critoone = crito[1]

endp
