; -----------------------------------------------------------------
; Library        cafpe
; -----------------------------------------------------------------
;  See_also     tp/cafpe/hsilv
; -----------------------------------------------------------------
;   Macro       hoptest
; -----------------------------------------------------------------
;   Keywords    nonparametric autoregression, bandwidth selection,
;               plug-in bandwidth, local linear estimation
; -----------------------------------------------------------------
;   Description Quantlet for computing the scalar plug-in bandwidth
;               for nonparametric estimation of nonlinear autoregression
;               models of order p. The unknown quantities in the
;               asymptotic optimal bandwidth are nonparametrically
;               estimated using C++ - routines via dlls.
;               Only the gaussian kernel can be used.
;
; -----------------------------------------------------------------
;   Usage   { Bhat,Bhatr,hB,Chat,sumwc,hC,hA } = hoptest(xsj,yorig,xtj,estimator,kernel,ntotal,sigy2,perB,robden)
;   Input
;      Parameter 
;      Definition
;	 Parameter ATTENTION:
;      Definition   this quantlet requires to open locling.dll, density.dll (NT)
;               or locling.so, denc.so (UNIX). This can be done with the quantlet cafpeload or 
;               directly with
;               garb = dlopen ("<path>\locling.dll") on NT,
;               garb = dlopen ("<path>\density.dll") on NT,
;               garb = dlopen ("<path>/locling.so") on UNIX,
;               garb = dlopen ("<path>/denc.so") on UNIX.

;       Parameter  xsj
;       Definition  (n x d) matrix of regressors
;       Parameter  yorig
;       Definition  (ny x 1) vector of dependent data
;       Parameter  xtj
;       Definition  (ny x d) matrix of lagged variables
;       Parameter  estimator
;       Definition  character variable for selection nonparametric estimator
;                   "loclin": local linear estimator; other estimators not
;                   implemented
;       Parameter  kernel
;    	Definition  character variable for selecting kernel for density estimation,
;                   must currently be "gaussian"
;       Parameter  ntotal
;       Definition  scalar, number of observations of available time series
;       Parameter  sigy2
;       Definition  scalar, variance of available time series
;       Parameter  perB
;       Definition  scalar, parameter screening off  0 <= perB < 1 observations
;                   with lowest density for estimating B,
;                   see also Section 5 in Tschernig & Yang (2000)
;       Parameter  robden
;       Definition  character variable for switching on/off robustification
;                   of density estimation a la Tjostheim & Auestad (1994),
;                   see also Section 5 in Tschernig & Yang (2000)
;                  "yes": on; "no": off
;   Output
;       Parameter  Bhat
;       Definition  scalar, estimated B, see eq. (3.2) in Tschernig & Yang (2000)
;       Parameter  Bhatr
;       Definition  scalar, number of observations used to estimate B after screening off 
;                   perB percent of the observations
;       Parameter  hB
;       Definition  scalar, predefined bandwidth for estimating B, if set to zero rule-of-thumb
;              	    bandwidth is computed, see eq. (5.1) in Tschernig & Yang (2000)
;       Parameter  Chat
;       Definition  scalar, estimated C using partial local quadratic estimation, 
;			see Section 5 in Tschernig & Yang (2000)
;       Parameter  Chatr
;       Definition  scalar, number of observations used to estimate C after screening off 
;                   perB percent of the observations   
;       Parameter  hC
;       Definition  scalar, predefined bandwidth for estimating C, if set to zero rule-of-thumb
;              	    bandwidth is computed, see Section 5 in Tschernig & Yang (2000) 
;       Parameter  hA
;       Definition  scalar, plug-in bandwidth for local linear estimation based on estimating
;                   the asymptotically optimal bandwidth, see Corollary 2.1 and Section 5 in 
;		    Tschernig & Yang (2000)

; -----------------------------------------------------------------
;   Reference Tschernig and Yang (2000), Nonparametric
;             Lag Selection for Time Series,
;             Journal of Time Series Analysis,
;             forthcoming  (http://www.blackwellpublishers.co.uk)
; -----------------------------------------------------------------
;   Author    Tschernig  000420
; -----------------------------------------------------------------


proc (Bhat,Bhatr,hB,Chat,Chatr,hC,hA) = hoptest(xsj,yorig,xtj,estimator,kernel,ntotal,sigy2,perB,robden);

; -------------------- initialize ---------------------------------

    d           = cols(xtj);
    dw          = d;
    nxest       = rows(xtj);
    sigK        = 1;
    if (kernel == "gaussian");
        Ksqint = 1 / (2*sqrt(pi))       ; kernel constant ||K||2^2  in TY
    endif;

; ------------------ estimate density p(x)     --------------------

     kernden    = "gaussian";
     looden     = 1;
     sig2x      = (prod( (var(xtj[,1:dw])') ))^(1/dw);
     hp         = hsilv(sqrt(sig2x),dw,rows(xtj),kernden)*matrix(dw,1);
     p          = dencp(xtj[,1:dw],hp,xtj[,1:dw],looden,kernden);


; ----------- compute hB by rule-of-thumb -------------------

    if (robden == "yes");    /* robustification only needed for density estimation */
        xsj     = xtj;
    endif;

    sigy    = sqrt(sigy2);
    hB      = hsilv(sqrt(sig2x),dw,ntotal,kernel);


    yadj        = yorig;
    xsjadj      = xsj;
    xtjadj      = xtj;

;  ----------- compute hC by rule-of-thumb -------------------

    ; rule of thumb for derivative  estimation TY, 2000

    hC  =  2* sqrt(sig2x) * ( 4/(dw+4) )^( 1/(dw+6) ) *   ntotal^( -1/(dw+6) );




; -------------- compute Bhat -------------------------------------
; ---------------- estimate mean functions  --------------------


    fandg   = 0;    /* only cond. mean  */
    loo     = 1;    /* leave-one out    */
    lorq    = 1;    /* local linear     */

    err     = 1;
    i       = 1;
    do

	{ mA,gsqA,denA,err } = fvllc(xsjadj,yadj,hB,xtjadj,kernel,lorq,fandg,loo);

        ghat       = mA[,1];
        error( (estimator != "loclin"), "quantlet fpepres: choose local linear estimator")
        if (err == 1)
            if (i > 30);
               "fpepres: iteration limit of 30 exceeded for computing";
               "         Bhat; current bandwidth hB= " hB;
               "          returns missing value for Bhat and Chat";
               garb = 0/0;
            else
                hB  = hB * 2;
           endif;
        endif
           i   = i + 1;

    until (err == 0);

; ---------------------- compute Bhat ----------------------------

        wB          = wei(p,perB,1);
        Bhatr       = sum(wB);
        res2        = (yadj - ghat)^2 .* wB;
        Bhat       = sum(res2 ./ p .* wB) / Bhatr;

; -------------- compute Chat -------------------------------------
; ---------------- estimate mean functions  --------------------

    err     = 1;
    i       = 1;
    do

        fandg   = 0;    /* only cond. mean  */
        loo     = 0;    /* leave-one out    */
        lorq    = 2;    /* local quadratic  */

        { mA,gsqA,denA,err } = fvllc(xsjadj,yadj,hC, xtjadj,kernel,lorq,fandg,loo);
        msecdir  = mA[,dw+2:cols(mA)];
        ghat       = mA[,1];

        if (err == 1)
            if (i > 30);
            "fpepres: iteration limit of 30 exceeded for computing";
            "          Chat; current bandwidth hC= " hC;
                "          returns missing value for Chat";
        garb   = 0/0;

            else;
            hC  = hC * 2;
            endif
         endif;
         i   = i + 1;

    until (err == 0)

; ------------------------- compute Chat ----------------------------

        wC          = wei(p,perB,1);
        Chattr    = zeros(nxest,1);
        i = 1;
        do
            Chattr[i] =  (  sum( (msecdir[i,])' )   )^2 *wC[i];
            i           = i+1;
        until (i > nxest);
        Chat       = sum( Chattr ) / nxest;

; ---------------------- compute hA -------------------------------

    if (abs(Chat) < 1e-10);
        hA = 100*(var(yorig))       ; take very large bandwidth
                                        ; in case Chat hints on linear model

    else;
    Ksq    = 2^(-0.5)*(prod(pdfn(0)'));
        hA  =  (dw*Ksq^dw * Bhat * rows(yadj)^(-1) * Chat^(-1) * 1 )^(1/(dw+4));
    endif;
;"Bhat" Bhat;
;"Bhatr" Bhatr;
;"hB" hB;
;"Chat" Chat;
;"sum(wC)" sum(wC);
;"hC" hC;
;"hA" hA;
;print yadj;
;print xtjadj;
;print xsjadj;

Chatr = sum(wC)
endp;
