; -----------------------------------------------------------------
; Library        cafpe
; -----------------------------------------------------------------
;  See_also	tp/cafpe/hoptest
; -----------------------------------------------------------------
;   Macro       hsilv
; -----------------------------------------------------------------
;   Keywords    nonparametric estimation, bandwidth selection, 
;               rule-of-thumb bandwidth
; -----------------------------------------------------------------
;   Description Quantlet to compute Silverman's rule-of-thumb bandwidth 
;               for density estimation using either the gaussian or uniform kernel
; -----------------------------------------------------------------
;   Usage   h = hsilv(sigy,d,ntotal,kerntype)
;   Input
;     Parameter  sigy
;     Definition scalar, standard deviation of regressors
;     Parameter  d
;     Definition number of continuous regressors
;     Parameter  ntotal
;     Definition number of observations for estimation
;     Parameter  kerntype
;     Definition type of kernel used:  "gaussian" or "quartic"
;   Output
;     Parameter  h
;     Definition  Silverman's rule-of-thumb bandwidth
; -----------------------------------------------------------------
;   Example     pathcafpe 	= "tp/cafpe/" 	; path of CAFPE quantlets
;		func(pathcafpe + "hsilv.xpl")	; load required XploRe files of CAFPE
;               d       	= 2
;               ntotal  	= 100
;               kerntype	= "gaussian"
;               y       	= normal(ntotal,d)
;               sigy    	= prod(var(y)')^(1/d)
;               h 		= hsilv(sigy,d,ntotal,kerntype)
;               "h" h
; -----------------------------------------------------------------
;   Result    returns Silverman's rule-of-thumb bandwidth
; -----------------------------------------------------------------
;   Reference Silverman, B. (1986), Density estimation for Statistics and Data
;               Analysis, Chapman and Hall, London.
; -----------------------------------------------------------------
;   Author    Tschernig  000420
; -----------------------------------------------------------------

proc (h) = hsilv(sigy,d,ntotal,kerntype);

    switch
    case (kerntype == "gaussian");
        sigK   = 1;
        Ksq    = 1/(2*sqrt(pi));
        break
    case (kerntype == "uniform");
        sigK   = 1/12;
        Ksq    = 1;
        break
    default
        error(1,"quantlet hsilv: chosen kernel type not available")
    endsw
    h   =   sigy * ( 4*(Ksq*2*sqrt(pi))^d/(sigK^4*(d+2)) )^( 1/(d+4) ) * ntotal^(-1/(d+4));

endp

/* ============================================================= */
