; -----------------------------------------------------------------
; Library        cafpe
; -----------------------------------------------------------------
;  See_also      tp/cafpe/cafpedefault, tp/cafpe/cafpe, tp/cafpe/cafpeload
; -----------------------------------------------------------------
;   Macro        plotloclin
; -----------------------------------------------------------------
;   Keywords     nonparametric time series analysis,
;                local linear regression, graphics
; -----------------------------------------------------------------
;   Description  Quantlet to compute for a given lag vector 
;			on given grid range a 1- or 2-dimensional plot of the regression
;                function of a nonlinear autoregressive process;
;                if more than 2 lags are used, then only two
;                lags are allowed to vary, the others have to be
;                fixed at values that are given by the user;
;                the procedure uses a plug-in bandwidth;
;                for this bandwidth the vector of residuals 
;                and the matrix of regressors are returned on which
;                the bandwidth estimation was conducted (for the default values
;                of the advanced parameters see the quantlet cafpedefault);
;		 	plots also standardized residuals
; -----------------------------------------------------------------
;   Reference    Tschernig and Yang (2000), Nonparametric
;                Lag Selection for Time Series,
;                Journal of Time Series Analysis,
;                forthcoming (http://www.blackwellpublishers.co.uk)
; -----------------------------------------------------------------
;   Usage {hplugin,hB,hC,xs,resid} = plotloclin(xdata,xresid,xdataln,xdatadif,xdatastand,volat,lags,h,xsconst,gridnum,gridmax,gridmin)
;   Input
;      Parameter 
;      Definition
;	 Parameter ATTENTION:
;      Definition         this quantlet requires to open locling.dll, density.dll (NT)
;               or locling.so, denc.so (UNIX). This can be done with the quantlet cafpeload or 
;               directly with
;               garb = dlopen ("<path>\locling.dll") on NT,
;               garb = dlopen ("<path>\density.dll") on NT,
;               garb = dlopen ("<path>/locling.so") on UNIX,
;               garb = dlopen ("<path>/denc.so") on UNIX.

;     Parameter   xdata
;     Definition     n x 1 matrix of the observed time series
;	Parameter	xresid
;	Definition	   n' x 1 vector of residuals from fitting conditional mean function
;     Parameter   xdataln
;     Definition     character variable: "yes": take natural logs, "no": don't
;     Parameter   xdatadif
;     Definition     character variable: "yes": take first differences of data, "no": don't
;     Parameter   xdatastand
;     Definition     character variable: "yes": standardize data, "no": don't
;	Parameter	volat
;     Definition  character variable: "no" plots conditional mean function;
;                   "resid" plots conditional volatility function,
;                   the residuals of fitting a conditional mean
;                   function have to be contained in xresid
;     Parameter   lags
;     Definition     m x 1 vector of lags
;	Parameter	h
;	Definition	scalar bandwidth for which if set to zero a scalar plug-in bandwidth using
;									hoptest is computed or a (d x 1) vector bandwidth
;     Parameter   xsconst
;     Definition     m x 1 vector (only needed if m>2)
;                    indicates which lags vary and which are kept
;                    fixed
;                    for those keeping fixed, the entry in the
;                    correponding row contains the value at which it
;                    is fixed
;                    for those to be varied, the entry in the
;                    corresponding row is 1e-100
;     Parameter   gridnum
;     Definition     scalar, number of grid points in one direction
;     Parameter   gridmax
;     Definition  scalar, maximum for grid
;     Parameter   gridmin
;     Definition  scalar, minimum for grid
;   Output
;       Parameter     hplugin
;       Definition    scalar, plug-in bandwidth or chosen scalar or vector bandwidth
;       Parameter     hB
;       Definition    scalar, rule-of-thumb bandwidth for nonparametrically
;                     estimating the constant B in CAFPE and for computing the
;                     plug-in bandwidth
;       Parameter     hC
;       Definition    scalar, rule-of-thumb bandwidth for nonparametrically
;                     estimating the constant C for computing the plug-in
;                     bandwidth
;       Parameter     xs
;       Definition    m x n' matrix with lagged values of time series which are
;                     used to compute plug-in bandwidth and residuals for
;                     potential diagnostics
;       Parameter     resid
;       Definition    vector with residuals after fitting a local linear
;                     regression at xs

; -----------------------------------------------------------------
;   Example    pathcafpe 	= "tp/cafpe/" ; path for CAFPE quantlets
;		   library("xplore")
;		   library("times")
;		   func(pathcafpe + "cafpeload"); load required XploRe files of CAFPE
;	         cafpeload(pathcafpe)    	
;              n              = 50

;           truedat    = "";
;           xresid     = 0
;           xdataln    = "no";
;           xdatadif   = "no";
;           xdatastand = "no";
;           lags       = 1|2
;	    h          = 0
;           xsconst    = 1e-100|1e-100
;           gridnum    = 10
;           gridmax    = 3
;	    gridmin    = -5
;           volat      = "no"
;
;           ; generate data
;           randomize(0)     ; sets seed value for random number generator
;           y=genexpar(1,0.1,0.3|0.6, 2.2|-0.8,normal(n))  ; generate exponential AR(2) process
;
;           ; plot generated data
;           title = "Realisation of exponential AR(2) process"
;           xlabel= "Time"
;           ylabel= "Values"
;           plotoneline(1:n~y,title,xlabel,ylabel)
;
;           ; compute plug-in bandwidth and plot regression function for given lags
;           { hplugin,hB,hC,xs,resid } = plotloclin(y,xresid,xdataln,xdatadif,xdatastand,volat,lags,h,xsconst,gridnum,gridmax,gridmin)

;           "plug-in bandwidth" hplugin
; 
; -----------------------------------------------------------------
;   Result    surface plot of regression function on a grid
;             determined by the data as well as used bandwidths
; -----------------------------------------------------------------
;   Author    Tschernig  000420
; -----------------------------------------------------------------


proc (hplugin,hB,hC,xs,resid) = plotloclin(xdata,xresid,xdataln,xdatadif,xdatastand,volat,lags,h,xsconst,gridnum,gridmax,gridmin)

; get default settings of advanced parameters
{ xresiddef,trueres,volatdef,startval,robden,selcrit,perA,perB,noutputf,outpath }=cafpedefault()
 
if (!(volat=="resid"))
	xresid = xresiddef
	volat	 = volatdef
endif

n       = rows(xdata);
m       = rows(lags);
lagmax  = max(lags)
dmax    = m

;"---------- Prior data transformations --------------------------- ";"";
if (volat == "resid")
    xresid      = xresid^2;
endif;

if (xdataln == "yes");
    if (min(xdata,2) > 0);
        xdata   = log(xdata);
    else;
        error(min(xdata,2)<0,"taking logarithms is not possible as there are negative observations");
    endif;
endif;
if (xdatadif == "yes");
    xdata   = xdata[2:rows(xdata)] - xdata[1:rows(xdata)-1];
endif;
if (xdatastand == "yes");
    xraw    = xdata ./ sqrt(var(xdata));
    if (rows(xresid) > 1);
        xresid  = xresid ./ sqrt(var(xresid));
    endif;
else;
    xraw    = xdata;
endif;
sig2hat = (var(xraw));


; compute optimal bandwidth for given lag vector


estimator   = "loclin"
kernel      = "gaussian"
Ksqint 	    = 1 / (2*sqrt(pi))       ; kernel constant ||K||2^2  in TY

{ crito,Ahat,sig2wnhat,hplugin,hB,hC,critgrid,xs,y,resid}=fpenpsl(xraw,xresid,lags,lagmax,h,volat,startval,robden,estimator,kernel,selcrit,perA,perB,Ksqint);

		

;bb  = inv(xs'xs)*xs'y;
; "Results of fpenpsl: variance y"
;var(y);

; plot standardized residuals
residstand = (resid-mean(resid)) / sqrt(var(resid))
title = "Standardized estimated residuals"
xlabel= "Time"
ylabel= "Standardized residuals"
xdresid = (1:rows(residstand))~residstand
dresid=createdisplay(1,1)
setmaskl (xdresid, (1:rows(xdresid))', 0, 1) 
show(dresid,1,1,xdresid)
setgopt(dresid, 1, 1,"title",title,"xlabel",xlabel,"ylabel",ylabel)


; prepare grid


xgridmax    = gridmax
xgridmin    = gridmin

xgridstep   = (xgridmax-xgridmin)/(gridnum-1);



switch
case  (m==1)
    xgrid2      = aseq(xgridmin,gridnum,xgridstep)
    break
case  (m==2)

    xgrid       = makegrid(xgridmin,xgridstep,gridnum);
    xgrid2      = xgrid
    break
case  (m>2)
    xgrid       = makegrid(xgridmin,xgridstep,gridnum);
    xgrid2      = matrix(rows(xgrid),m)
    mcount      = 1
    xgridcount  = 1
    do
        if (xsconst[mcount]==1e-100)
            xgrid2[,mcount] = xgrid[,xgridcount]
            xgridcount      = xgridcount + 1
        else
            xgrid2[,mcount] = xsconst[mcount] * xgrid2[,mcount]
        endif
        mcount = mcount + 1
    until (mcount > m)  
    break
endsw
"maximum of regressors" max(xs)
"minimum of regressors" min(xs)
"maximum of grid" max(xgrid2)
"minimum of grid" min(xgrid2)

{ ghat,denA,err } =  fgrfsv(xs,y,hplugin,xgrid2,kernel,estimator);

"maximum of function estimate" max(ghat)
"minimum of function estimate" min(ghat)

if (volat == "resid")
	title = "Estimated conditional standard deviation"
else
        title = "Estimated conditional mean function"
endif	
switch
case (m==1)
   d1 	= createdisplay(1,1)	    	
   xx 	= xgrid2~ghat
   setmaskl (xx, (1:rows(xx))', 0, 1) 
   show (d1, 1, 1, xx)

   setgopt(d1, 1, 1,"title",title,"xlabel","Regressor","ylabel","Estimated Values")
   break

case (m==2)

    ; generate the matrix that tells setmaskl how to connect the cornerpoints
    box=grid(min(xgrid~ghat)',max(xgrid~ghat)'-min(xgrid~ghat)',2|2|2)
    cube=(1~2~4~3~1)|(5~6~8~7~5)|(1~5~0~0~0)|(2~6~0~0~0)|(3~7~0~0~0)|(4~8~0~0~0)
    setmaskl(box,cube,0,1,1)
    boxtext = " "|"Lag 1 "|"Lag 2 "|" "|" "|"f "|" "|" "
    boxtpos = 6|6|9|-1|-1|9|-1|-1
    setmaskt(box,boxtext,0,boxtpos,26)

  d1  = createdisplay(1,1)

  gs1 = grsurface(xgrid~ghat)
  show (d1, 1, 1, box,gs1)
  setgopt(d1, 1, 1,"title",title)
  break

case (m>2)

    ; generate the matrix that tells setmaskl how to connect the cornerpoints
    box=grid(min(xgrid~ghat)',max(xgrid~ghat)'-min(xgrid~ghat)',2|2|2)
    cube=(1~2~4~3~1)|(5~6~8~7~5)|(1~5~0~0~0)|(2~6~0~0~0)|(3~7~0~0~0)|(4~8~0~0~0)
    setmaskl(box,cube,0,1,1)
    boxtext = " "|"1st regressor "|"2nd regressor "|" "|" "|"f "|" "|" "
    boxtpos = 6|6|9|-1|-1|9|-1|-1
    setmaskt(box,boxtext,0,boxtpos,26)
    gs1 = grsurface(xgrid~ghat)
    d1  = createdisplay(1,1)
    show (d1, 1, 1, box,gs1)
    setgopt(d1, 1, 1,"title",title)
  break

endsw


;wgr         = gridsel(xgrid,denA,xs[.,1 2],xgridstep,pergr);
                        /* CHANGE */

endp
