proc(a, b, xs)=wavesmo(x, l, h, s, tt)
; -----------------------------------------------------------------------
; Library      wavelet
; -----------------------------------------------------------------------
; See_also     fwt invfwt fwt2 invfwt2 fwttin
; -----------------------------------------------------------------------
; Link         ../tutorials/wavestart.html Wavelet tutorial
; -----------------------------------------------------------------------
; Macro        wavesmo
; -----------------------------------------------------------------------
; Description  Generates smoothed mother wavelet coeffients and the
;              resulting estimate. x contains the vector of data and l
;              gives the number of father wavelet coefficient (power of 
;              2).
;              h is the vector of wavelet basis coeffients (automatically
;              generated by calling the quantlib wavelet). s contains the threshold 
;              values, either one for all mother wavelet coefficients or 
;              for each level. Set tt=0 for hard thresholding and tt=1
;              for soft thresholding.
; -----------------------------------------------------------------------
; Usage        {a, b, xs} = wavesmo (x, l, h, s, tt)
; Input
;   Parameter  x 
;   Definition n x 1 array (n has to be a power of 2), input data, 
;                     observed at an equispaced grid
;   Parameter  l 
;   Definition p x 1 array, giving the number of father wavelets  
;   Parameter  h
;   Definition q x 1 array, wavelet basis coefficients
;   Parameter  s 
;   Definition r x 1 array, binwidth   
;   Parameter  tt
;   Definition scalar, 0=hard, 1=soft thresholding     
; Output
;   Parameter  a
;   Definition l   x 1 array, father wavelet coefficients   
;   Parameter  b
;   Definition n-l x 1 array, mother wavelet coefficients  
;   Parameter  xs
;   Definition n   x 1 array, smoothed estimate of x   
; -----------------------------------------------------------------------
; Example      ; loads the library wavelet
;              library ("wavelet")
;              ; generates an equidistant grid from 0 to 1.5 with
;              ; binwidth 0.1 
;              x = grid(0, 0.1, 16)
;              ; computes z_i = x_i *x_i
;              z = x^2
;              ; chooses the number father wavelet as one
;              l = 1
;              ; computes the thresholded coefficients and
;              ; the thresholded data
;              {a, b, xs} = wavesmo (z, l, haar, 0.1, 0)
;              a
;              b
;              xs 
; -----------------------------------------------------------------------
; Result       Unthresholded father wavelet coeffients a and thresholded
;              mother wavelet coefficients b and smoothed estimate xs 
; -----------------------------------------------------------------------
; Author       Sigbert Klinke, 960618, Sperlich 980105
; -----------------------------------------------------------------------
  error (dim(dim(s)).<>1, "WAVESMO : s vector or scalar")
  {a,b} = fwt(x, l, h)
  if (exist(tt).<=0)
    tt = 0                        ; assume hard threshold     
  endif
  b3 = abs(b[,3])  
  if (dim(s).=1)                  ; s scalar
    ss = s.*matrix(rows(b))
  else                            ; s vector
    b1 = b[,1]
    b1 = b1-min(b1)+1
    if (rows(s).<max(b1))
      s = s|(matrix(max(b1)-rows(s)).*(1+max(b3)))
    endif
    ss = index(s, b1)
    ss
  endif
  cb = b3.<ss
  if (tt)                         ; tt=1 soft threshold
    bs    = 2.*((b[,3].>0)-0.5).*cb
    b[,3] = b[,3].*cb 
    b[,3] = b[,3] - bs .* ss   
  else                            ; tt=0 hard threshold
    b[,3] = b[,3].*cb 
  endif
  xs = invfwt (a, b, rows(x), l, h)
endp
