proc(summ) = wavetest(x, y, p, dis, levels, data, adjust) 
; ----------------------------------------------------------------- 
; Library        gam 
; ----------------------------------------------------------------- 
;  See_also      intest, intestpl, gintest, gintestpl,backfit 
; ----------------------------------------------------------------- 
;   Macro        wavetest 
; ----------------------------------------------------------------- 
;   Description   Additive component analysis in additive separable models  
;                 using wavelet estimation. The first (additive) component  
;                 is tested against a given polynomial form of degree p,  
;                 e.g., if p=1 is to test linearity, p=0 is to test for  
;                 significant influence of the first component at all etc.  
;                 For details see Haerdle, Sperlich, Spokoiny (1997), DP  
;                 52, SFB 373 Berlin.  
; ----------------------------------------------------------------- 
;   Usage         erg =  wavetest(x,y,p{,dis{,data{,adjust}}}}) 
;   Input 
;     Parameter   x 
;     Definition      n x d matrix , the observed explanatory variable where  
;                     all dummy inputs (0-1 variables) are expected to be in 
;                     the last ('dis') columns. 
;     Parameter   y 
;     Definition      n x 1 matrix , the observed response variable 
;     Parameter   p 
;     Definition      scalar , the polynomial degree of the Hypothesis, 
;                     see Description. 
;     Parameter   dis 
;     Definition      scalar , the number of dummy variables in x.  
;                     Have to be the last columns in x. Default is dis=0. 
;     Parameter   data 
;     Definition      optional string, if output file for the wavelet  
;                     coefficient estimates is wished, this string will be  
;                     the name of that output file. 
;     Parameter   adjust 
;     Definition      optional 2 x 1 vector, multiplicators for the critical 
;                     values. E.g. adjust=(0.5, 1) give about 5% first kind  
;                     error, adj=(1, 2.5) give about 1%. Default is (1, 1). 
;                     For details see Haerdle, Sperlich, Spokoiny (1997).  
; 
;   Output 
;     Parameter   erg 
;     Definition    object of string, the table of results 
;     Parameter   data 
;     Definition    data file , containing the wavelet estimates  
; ----------------------------------------------------------------- 
; Notes  Data file "data" with wavelet estimates will generated. Note that a logfile  
;        "prot.log" will generated with some detailed information about the 
;        generation process.  
; ----------------------------------------------------------------- 
;   Example   library("gam") 
;             randomize(1234) 
;             n   = 100            ; sample size 
;             x   = normal(n,3)    ; regressor variables 
;             eps = normal(n,1) * sqrt(0.8)    ; error term 
;             y   = sin(2*x[,1]) + x[,2]^2 + 2*x[,3] +eps   ; model  
;                ;  first component of the model is sin-function 
;             p   = 1              ;  H_0: first component is linear 
;             erg = wavetest(x,y,p)   ; no dis, as dis=0 is already default 
;             erg                     ; show table of results 
; ----------------------------------------------------------------- 
;   Result    Contents of erg
;             
;             [ 1,] " "
;             [ 2,] "-------------------------------------------------------" 
;             [ 3,] "no output of function estimates" 
;             [ 4,] "-------------------------------------------------------" 
;             [ 5,] "HYPOTHESIS: 1.add.component is polynomial of degree  1" 
;             [ 6,] "-------------------------------------------------------" 
;             [ 7,] "Hypothesis has been rejected at level  0" 
;             [ 8,] "-------------------------------------------------------" 
;             [ 9,] "highest possible level is  3" 
;             [10,] " " 
;             [11,] "-------------------------------------------------------" 
;             [12,] "-------------------------------------------------------" 
;             [13,] "            local Test              chi-2 like Test " 
;             [14,] "- - - - - - - - - - - - - - - - - - - - - - - - - - - -" 
;             [15,] "level  crit.value  test stat.    crit.value  test stat. " 
;             [16,] "-------------------------------------------------------" 
;             [17,] "  0      2.74767      4.34917      3.24940     12.66804" 
;             [18,] "-------------------------------------------------------" 
;             [19,] "-------------------------------------------------------" 
;             [20,] " "
; ----------------------------------------------------------------- 
;   Author    Sperlich 970730 
; ----------------------------------------------------------------- 
; 
d   = cols(x) 
n   = rows(x); 
if (exist(dis)==0) 
  dis = 0 
endif 
x12 = x[,1:(d-dis)] 
if (dis>0) 
  x3 = x[,(d-dis+1):d] 
endif 
aloc = 1.0  ; 
adet = 1.0 ; 
if (exist(adjust)) 
  aloc = adjust[1] 
  adet = adjust[2] 
endif 
; 
jj= floor((log(n/(d-dis) -dis-p)/log(2) -1)); 
jn = matrix(d)*0 
jn[1:(d-dis)] = jj 
; 
nn = matrix(d-dis,2) 
j=0 
 while (j<(d-dis)) 
  j = j+1 
  {xr,yr} = discrete(x12[,j]) 
  nn[j,1] = rows(xr) 
  nn[j,2] = rows(xr)<(2^(jj+1) -1) 
  if (nn[j,2]) 
    jn[j] = floor(log(nn[j,1])/log(2)-1) 
  endif 
 endo 
; 
while ((n-sum(2^(jn+1))-d+p)<(n/4)) ; 
  jn = replace(jn,jj,(jj-1)) ; 
  jj = jj -1 ; 
endo ; 
; 
; studentize the continuous data but without correlation correction 
  x = x12 ./ sqrt( var(x12) ) 
  if (dis.>0) 
    x = x12~x3 
  endif 
  x = (x-min(x)) ./ (max(x)-min(x)) ; 
; 
/**************************************************************/ 
/********** Pre-estimation to get a variance estimate *********/ 
/**************************************************************/ 
; 
/*****************  Haarbasis construction  ********************/ 
 j = 1; 
 mult = 1 ; 
 subt = 0 ; 
 do ; 
   mult = mult | (matrix(2^j,1) * 2^j); 
   subt = subt | aseq(0,2^j,1) ; 
   j    = j+1 ; 
 until (j>jn[1]) ; 
; 
  xd1 = matrix(n) 
  j = 0 
  while (j<p) 
    j = j+1 
    xd1 = xd1~(x[,1]^j) ; 
  endo 
  tra = (mult'.* x[,1]) - subt' ; 
  xd1 = xd1~( ((0.5.<tra).*(tra.<=1))-((0.<tra).*(tra.<=0.5)) ); 
  lam1= sqrt( sum(xd1^2) ) ; 
  lam1= (abs(lam1).<0.0001)+(abs(lam1).>=0.0001).*lam1 ; 
  lam1= lam1^(-1) ; 
  xd1 = lam1 .* xd1 ; 
 ; 
 XD = matrix(n)*0 ; 
 k=2 ; 
 do ; 
 /* 
   if (nn[j,2]) 
     j = 1 
     mult2 = 1 ; 
     subt2 = 0 ; 
     do 
       mult2 = mult2 | (matrix(2^j,1) * 2^j) ; 
       subt2 = subt2 | aseq(0,2^j,1) ; 
       j    = j+1 ; 
     until (j>jn[k]) ; 
   else 
     mult2 = mult 
     subt2 = subt 
   endif 
   tra = (mult2'.* x[.,k]) - subt2' ; 
   XDt = ( ((0.5.<tra).*(tra.<=1))-((0.<tra).*(tra.<=0.5)) ); 
   XDt = sqrt(mult2').*XDt ; 
   XD  = XD~XDt ; 
 */ 
   tra = (mult[1:(2^(jn[k]+1)-1)]'.* x[,k]) - subt[1:(2^(jn[k]+1)-1)]' ; 
   XDt = ( ((0.5.<tra).*(tra.<=1))-((0.<tra).*(tra.<=0.5)) ); 
   XDt = sqrt(mult[1:(2^(jn[k]+1)-1)]').*XDt ; 
   XD  = XD~XDt ; 
   k = k+1; 
 until (k>(d-dis)) ; 
 ; 
 if (dis>0) 
  k = d-dis+1 ; 
  do ; 
    XDt = ( ((0.5.<x[,k]).*(tra.<=1))-((0.<tra).*(tra.<=0.5)) ); 
    XDt = sqrt(mult[1:(2^(jn[k]+1)-1)]').*XDt ; 
    XD  = XD~XDt ; 
    k = k+1; 
  until (k>d)  ; 
 endif 
 ; 
 dim = cols(XD) ; 
 XDv= XD[,2:dim] ; 
 XD = xd1~XDv ; 
 dim = dim-1 ; 
 dim1= cols(xd1) ; 
  N0    = 2^(jn+1)-1 ; 
  N0[1] = N0[1]+1+p ;            = dim1 if correct 
  NN    = sum(N0);               = dim  if correct 
/***********  pre-estimation routine and saving  **************/ 
 xdxd = XD'*XD ; 
 ixtx = ginv(XD'*XD) 
 bb   = ixtx * XD' *y ; 
 epss = y - XD*bb ; 
 capsig= epss'*epss ; 
 vari  = capsig / (n-sum(xdiag(ixtx*xdxd))) ; 
 ; 
 if (exist(data)) 
  an = 1 ; 
  en = N0[1] ; 
  j=1; 
  do 
    fh = XD[,an:en]*bb[an:en,] ; 
    write(fh,data) 
    an = an +N0[j]  ; 
    j=j+1 ; 
    en = en +N0[j] 
  until (j>(d-dis)) 
 endif 
 /******************    ****************************/ 
 /***********      Testing procedure       *********/ 
 /**************************************************/ 
 threshloc = matrix(jn[1]+1) 
 threshdet = matrix(jn[1]+1) 
 Tloc = matrix(jn[1]+1) 
 Tdet = matrix(jn[1]+1) 
 ; 
 lev = jn[1] 
 j1= 0 ; 
 do ; 
  /************  Haarbasis construction  ********************/ 
   xd1 = matrix(n) 
   j = 0 
   while (j<p) 
     j = j+1 
     xd1 = xd1~(x[,1]^j) ; 
   endo 
   tra = (mult[1:(2^(j1+1)-1)]'.* x[,1]) - subt[1:(2^(j1+1)-1)]' ; 
   xd1 = xd1~( ((0.5.<tra).*(tra.<=1))-((0.<tra).*(tra.<=0.5)) ); 
    lam1= sqrt( sum(xd1^2) ) ; 
    lam1= (abs(lam1).<0.0001)+(abs(lam1).>=0.0001).*lam1 ; 
    lam1= lam1^(-1) ; 
   xd1 = lam1 .* xd1 ; 
  ; 
     XD = xd1~XDv ; 
    dim1= cols(xd1) ; 
  ; 
   N0    = 2^(jn+1)-1 ; 
   N0[1] = N0[1]+1+p   ;         = dim1 if correct 
   NN    = sum(N0)     ;         = dim  if correct 
  ; 
  ixtx = ginv(XD'*XD) 
  bb  = ixtx * XD' *y ; 
  /***************  test statistic  ***************/ 
  j2 = j1+1 
  threshloc[j2] = 1+sqrt(2*log(2^j1)+aloc*2*log(log(n))) 
  threshdet[j2] = 1/sqrt(2)+sqrt(2^j1/(2*n))+adet*sqrt(4*log(log(n))) 
  ; 
  Vd   = ixtx[(dim1-2^(j1)+1):dim1,(dim1-2^(j1)+1):dim1] ; 
  bbd  = bb[(dim1-2^(j1)+1):dim1,1] ; 
  dim1 = rows(bbd) ; 
  ; 
  dia = vari*xdiag(Vd) ; 
  dia = (abs(dia).<0.00001)+(abs(dia).>=0.00001).*dia ; 
  Tlocall = abs( bbd ) ./ sqrt(dia) ; 
  Tloc[j2] = max( Tlocall ); 
  tr  = sum(xdiag((ginv(Vd)*Vd))); 
  Tdet[j2] = ( (bbd'*ginv(Vd)*bbd)/vari -tr) / sqrt(2*tr) 
   ; 
  philoc = Tloc[j2] > threshloc[j2] ; 
  phidet = Tdet[j2] > threshdet[j2] ; 
  phi    = max(philoc|phidet) ; 
  if (phi) ; 
    lev = j1 
    j1 = jn[1]; 
  endif; 
  ; 
  j1 = j1+1; 
 until (j1>jn[1]) ; 
 ; 
 if (exist(data)) 
   text1 =("function estimates has been written in ")+data 
 else 
   text1 ="no output of function estimates" 
 endif 
 text2 ="HYPOTHESIS: 1.add.component is polynomial of degree " 
 text2 = text2+string("%2.0f",p) 
 if (phi) 
   text3 ="Hypothesis has been rejected at level " 
   text3 = text3+string("%2.0f",lev) 
 else 
   text3 ="Hypothesis has not been rejected" 
 endif 
 text4 =("highest possible level is ")+string("%2.0f",jn[1]) 
 vec1 = ("      ")+("      local Test      ") 
 vec1 = vec1 +("        chi-2 like Test ") 
 vec2 = ("level ")+(" crit.value ")+(" test stat.  ") 
 vec2 = vec2+("  crit.value ")+(" test stat. ") 
 hline ="-------------------------------------------------------" 
 hlined="- - - - - - - - - - - - - - - - - - - - - - - - - - - -" 
 lev1 = lev+1 
 mat = string("%3.0f",(0:lev)) 
 mat = mat + string("%13.5f",threshloc[1:lev1]) 
 mat = mat + string("%13.5f",Tloc[1:lev1]) 
 mat = mat + string("%13.5f",threshdet[1:lev1]) 
 mat = mat + string("%13.5f",Tdet[1:lev1]) 
 summ = " "|hline|text1|hline|text2|hline|text3|hline|text4 
 summ = summ|" "|hline|hline|vec1|hlined|vec2|hline 
 summ = summ|mat|hline|hline|" " 
 ; 
endp 
