proc(fh, clo, cup) = dencb(x,h,alpha,K,d) 
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      denest denci denxest denxci denxcb  
; -----------------------------------------------------------------
;   Macro        dencb
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel density estimation, 
;                confidence bands
; -----------------------------------------------------------------
;   Description  computes uniform confidence bands with 
;                prespecified confidence level for univariate 
;                density estimation.    
; -----------------------------------------------------------------
;   Reference    Haerdle (1990): Applied Nonparametric Regression
; 
;                Haerdle (1991): Smoothing Techniques
; -----------------------------------------------------------------
;   Usage        {fh, clo, cup} = dencb(x {,h {,alpha {,K} {,d}}})
;   Input
;     Parameter  x  
;     Definition   n x 1 vector, the data.
;     Parameter  h 
;     Definition   scalar, bandwidth. If not given, the rule of
;                  thumb bandwidth computed by denrot is used 
;                  (Silverman's rule of thumb).
;     Parameter  alpha  
;     Definition   confidence level, If not given, 0.05 is used.
;     Parameter  K  
;     Definition   string, kernel function on [-1,1] with
;                  K(-1)=K(1)=0. If not given, the Quartic kernel 
;                  "qua" is used.
;     Parameter  d  
;     Definition   scalar, discretization binwidth. d must be smaller
;                  than h. If not given, the minimum of h/3 and 
;                  (max(x)-min(x))/100 is used. 
;   Output
;     Parameter  fh  
;     Definition   m x 2 matrix, the first column is a grid and the 
;                  second column contains the density estimate on 
;                  that grid.
;     Parameter  clo  
;     Definition   m x 2 matrix, the first column is a grid and the 
;                  second column contains the lower confidence 
;                  bounds for that grid.
;     Parameter  cup  
;     Definition   m x 2 matrix, the first column is a grid and the 
;                  second column contains the upper confidence 
;                  bounds for that grid.
; -----------------------------------------------------------------
;   Example   library("smoother")                                       
;             library("plot")                                       
;             x = 5*normal(200)+10
;             {fh, clo, cup} = dencb(x,3)
;             ;
;             fh=setmask(fh,"line","blue")
;             clo=setmask(clo,"line","blue","thin","dashed")
;             cup=setmask(cup,"line","blue","thin","dashed")
;             plot(fh,clo,cup)
; -----------------------------------------------------------------
;   Result    Confidence intervals at confidence level 
;             alpha = 0.05 for a normal density from N(10,25)  
;             are pictured using Quartic kernel (default) and 
;             bandwidth h=3.
; -----------------------------------------------------------------
;   Author    Lijian Yang, 980108; Marlene Mueller 990413
; -----------------------------------------------------------------
;
  error(cols(x)<>1,"dencb: cols(x) <> 1")
; 
  n=rows(x)
  if (exist(h)==0)
    h=denrot(x); h=2.78*sqrt(var(x))*(n^(-0.2))
  endif
  if (exist(alpha)==0)
    alpha = 0.05   ; default confidence level is 0.05    
  endif   
;
  if (exist(K)==0)
    cK=5/7
    c2=1.5
    K="qua"
  endif
  if (exist(K)==1) ; no K but d
    d=K
    cK=5/7
    c2=1.5
    K="qua"
  endif
  if ((exist(K)==2)&&(K!="qua"))
    {dK,cK,mK}=canbw(K)
    {dK,cK,mK}=canbw(K)
    ng=1000
    hg=2/(ng-1)
    g=grid(-1,hg,ng)
    g1=g[1:(rows(g)-1)]
    g2=g[2:rows(g)]
    DK=(_K(g2)-_K(g1))./hg  ; approx. derivative of K
    c1K=sum(DK^2)*hg        ; approx. norm of K'
    c2=c1K/(2*cK)
  endif
;
  if (exist(d)==0)
    d=(max(x)-min(x))/100
    d=min(d~(h./3),2)
  endif
;
  mx= min(x)
  rx= max(x)-min(x)
;
  x = (x-mx)./rx
  d = d./rx
  h = h./rx
  fh = denest(x,h,K,d)
  r = sqrt(-2*log(h))
  dn = r + 0.5*log( c2/(2*(pi^2)) )/r
  calpha = -log(-log(1-alpha)/2)
;  
  mrg = (calpha/r + dn) .* sqrt(cK.*fh[,2]/(n*h))
  clo = (rx.*fh[,1] + mx)~((fh[,2] - mrg)/rx)
  cup = (rx.*fh[,1] + mx)~((fh[,2] + mrg)/rx)
  fh  = (rx.*fh[,1] + mx)~(fh[,2]/rx)
endp  

