proc(m)=fastint(x, y, h1, h2, loc, xg)
; -----------------------------------------------------------------
;   Library      gam
; -----------------------------------------------------------------
;   See_also     intest, backfit 
; -----------------------------------------------------------------
;   Macro        fastint
; -----------------------------------------------------------------
;   Description  fastint estimates the additive components and their
;                derivatives of an additive model using a modification
;                of the integration estimator plus a one step backfit,
;                see Kim, Linton and Hengartner (1997) and Linton (1996) 
; -----------------------------------------------------------------
;   Usage         m = fastint(x,y,h1,h2,loc{,xg})   
;   Input
;     Parameter   x  
;     Definition      n x p matrix, the observed continuous 
;                     explanatory variable, see also xg.
;     Parameter   y 
;     Definition      n x q matrix, the observed response variables  
;     Parameter   h1
;     Definition      p x 1 vector or scalar, bandwidth for the pilot
;                     estimator. It is recommended to undersmooth here.
;     Parameter   h2
;     Definition      pg x 1 vector or scalar, bandwidth for the backfit 
;                     step. Here you should smooth in an optimal way.
;     Parameter   loc
;     Definition      {0,1,2}, degree of the local polynomial smoother used
;                     in the backfit step:
;                     0 for Nadaraya Watson, 1 local linear, 2 local quadratic
;     Parameter   xg
;     Definition      ng x pg matrix, optional, the points on which the
;                     estimates shall be calculated. the columns of t and
;                     tg must have the same order up to column pg < = p.
;                     If grid is used, the results won t get centered!
;   Output                                                           
;     Parameter   m
;     Definition      ng x pp matrix, where pp is pg*(loc+1). Estimates 
;                     of the additive functions in column 1 to pg, the first
;                     derivatives in column (pg+1) to (2*pg) and the second
;                     derivatives in column (2*pg+1) to (3*pg). 
; -----------------------------------------------------------------
;   Example   library("gam")
;             randomize(1234)
;             n = 100
;             d = 2
;            ; generate a correlated design:
;             var = 1.0
;             cov = 0.4  *(matrix(d,d)-unit(d)) + unit(d)*var
;             {eval, evec} = eigsm(cov) 
;             t = normal(n,d) 
;             t = t*((evec.*sqrt(eval)')*evec') 
;             g1    = 2*t[,1]
;             g2    = t[,2]^2 -mean(t[,2]^2)
;             y     = g1 + g2  + normal(n,1) * sqrt(0.5)
;             h1    = 0.5          
;             h2    = 0.7     
;             loc   = 0
;             gest  = fastint(t,y,h1,h2,loc)
;             library("graphic")
;             pic   = createdisplay(1,2)
;             dat11 = t[,1]~g1
;             dat12 = t[,1]~gest[,1]
;             dat21 = t[,2]~g2
;             dat22 = t[,2]~gest[,2]
;             setmaskp(dat12,4,4,8)
;             setmaskp(dat22,4,4,8)
;             show(pic,1,2,dat11,dat12)
;             show(pic,1,1,dat21,dat22)
; -----------------------------------------------------------------
;   Result    estimates of the additive functions 
; -----------------------------------------------------------------
;   Author    Hengartner, Haerdle, Sperlich 970901
; -----------------------------------------------------------------
  d  = cols(x)
  n  = rows(x)
  q  = cols(y) 
  if (rows(h1)<d)
    errtext = "rows of h1 didn't fit: "+string("%1.4f",h1[1,1])+" assumed"
    warning(rows(h1)>1,errtext)
    h1 = h1[1,1]*matrix(d)
  endif
  if (rows(h2)<d)
    errtext = "rows of h2 didn't fit: "+string("%1.4f",h2[1,1])+" assumed"
    warning(rows(h2)>1,errtext)
    h2 = h2[1,1]*matrix(d)
  endif
  ind= 1:n
  mh = matrix(n,d,q)
  j  = 1
  while (j<=d)
     ; loop over pilots
      s = x[,j]   
     ; index for coordinates
      indj = matrix(d)
      indj[j,] = 0
      t = paf(x', indj)'
      h10 = paf(h1,indj)
      indt = sort(t[,1]~ind)
      mhh = sker(t[indt[,2],],h10',"qua",matrix(n)) 
      fh2 = mhh[indt[,2],]  
     ; density of other components
      indx = sort(x[,1]~ind)
      mhh = sker(x[indx[,2],],h1',"qua",matrix(n)) 
      fh1 = mhh[indx[,2],]
      z = y.* fh2./fh1
     ; now smooth s against z
      inds = sort(s~ind)
      xs = sort(s~z)
      mhh = sker(xs[,1],h1[j],"qua",matrix(n)~xs[,2:(q+1)])
      mh[inds[,2],j,] = reshape(mhh[,2:cols(mhh)],(n|1|q))
      j = j+1
  endo
  if (exist(xg))
    ng = rows(xg)
    dg = cols(xg)
    ind = 1:ng
  else
    ng = n
    dg = d
    xg = x
  endif
  m = matrix(ng,dg*(loc+1),q) 
  y = reshape(y,(n|1|q))
  j = 1
  if (loc==0)
   while (j.<=dg)  
     z = y .- sum(mh,2) + mh[,j,]
     z = reshape(z,(n|q))
     ts = sort(x[,j]~z) 
     tg = sort(xg[,j]~ind)            
     rm = sker(ts[,1],h2[j],"qua",ts[,2:(q+1)]~matrix(n),tg[,1])
     mhh = rm[,1:q]./rm[,(q+1)]
     mhh = sort(tg[,2]~mhh,1)[,2:(q+1)]  
     m[,j,] = reshape(mhh,(ng|1|q))
     if (1-exist(xg)) 
       m[,j,] = replace(m[,j,],NaN,0)     
       m[,j,] = m[,j,] .- mean(m[,j,])
     endif
     j = j+1
   endo
  else
   while (j.<=dg)
     z = y .- sum(mh,2) + mh[,j,]
     z = reshape(z,(n|q))
     ts = sort(x[,j]~z)    
     tg = sort(xg[,j]~ind)            
     ma = locpol(ts[,1],tg[,1],ts[,2:(q+1)],h2[j],loc-1)  
     ma = sort(tg[,2]~ma)[,2:(cols(ma)+1),]
     m[,j,] = replace(ma,NaN,0)
     if (1-exist(xg))      
       m[,j,] = m[,j,] - mean(m[,j,])
     endif
     m[,j+dg,] = ma[,2,]
     if (loc==2)
       m[,j+2*dg,] = ma[,3,]*2
     endif  
     j = j+1
    endo
  endif
endp     
     
