proc(freq)=frequency(x,xvars,owidth)
; ----------------------------------------------------------------------------
; Library       stats
; ----------------------------------------------------------------------------
;  See_also     discrete descriptive cumsum
; ----------------------------------------------------------------------------
;   Macro       frequency
; ----------------------------------------------------------------------------
;   Description  provides frequency tables for all columns of a 
;                matrix. An additional vector of name  strings can
;                be given to identify columns by names.
; ----------------------------------------------------------------------------
;   Usage        freq=frequency(x{,xvars{,outwidth})
;   Input
;     Parameter   x  
;     Definition        n x p  matrix, the columns correspond to the 
;                       variables. The matrix may be numeric or string.
;     Parameter   xvars
;     Definition        optional, p x 1  string vector, variable names. 
;     Parameter   outwidth
;     Definition        optional, integer<71, outwidth for categories,
;                       default=15.
;   Output
;     Parameter   freq
;     Definition        string output, containing frequency table(s).
; ----------------------------------------------------------------------------
;   Example   library("stats")
;             setenv("outputstringformat", "%s") 
;             x=read("lizard")
;             xvars="height"|"diameter"|"shade"|"daytime"|"grahami"|"total"
;             frequency(x,xvars)
; ----------------------------------------------------------------------------
;   Result    frequency tables for all six variables, as e.g. for height:
;             [  1,]  
;             [  2,] ==================================================
;             [  3,]  Variable height
;             [  4,] ==================================================
;             [  5,]                 |  Frequency  Percent  Cumulative
;             [  6,] --------------------------------------------------
;             [  7,]               0 |         12    0.500      0.500
;             [  8,]               1 |         12    0.500      1.000
;             [  9,] --------------------------------------------------
;             [ 10,]                 |         24    1.000
;             [ 11,] ==================================================
;             [ 12,]  
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 970702
; ----------------------------------------------------------------------------
  if (exist(x)==9)
    x=x{1}
  endif
  if (exist(xvars)>0)
    if (cols(xvars)>1)
      xvars=reshape(xvars, cols(xvars)|rows(xvars))
    endif
  endif
  if (exist(owidth)==0)
    owidth=15
  else
    owidth=2.*(owidth.<2)+owidth.*(owidth.>=2).*(owidth.<71)+70.*(owidth.>=71)
  endif
;   
  error(rows(dim(x))>2,"input must be vector or matrix")
  error((exist(x)<1)||(exist(x)>2),"input must be string or numeric")
  if (exist(x)==1)
    x=(0*matrix(rows(x),cols(x))).*(x==0)+x.*(x!=0)
  endif
;
  if (exist(xvars)>0)
    error(exist(xvars)!=2,"variable names should be strings!")
    error(rows(dim(xvars))>1,"variable names have wrong dimension")
    error(rows(xvars)!=cols(x),"variable names have wrong dimension")
  else
    xvars=string("%1.0f",1:cols(x))
  endif
;
  freq=" "
;
  ostring1="======================================================================"
  ostring2="----------------------------------------------------------------------"
  ostring3="                                             "
  ostring3=ostring3+"                                    "
  ostring1=substr(ostring1,1,owidth+1)
  ostring2=substr(ostring2,1,owidth+1)
  ostring3=substr(ostring3,1,owidth+1)
  j=0
  while (j<cols(x))
    j=j+1
    {xr,r}=discrete(x[,j])
    freq=freq|ostring1+"==================================="
    freq=freq|" Variable "+xvars[j]
    freq=freq|ostring1+"==================================="
    freq=freq|ostring3+" |  Frequency  Percent  Cumulative "
    freq=freq|ostring2+"-----------------------------------"
    if (exist(xr)==1)
      form="%"+string("%1.0f",owidth)+".8g"
      cont=string(form,xr)
    else
      form="%"+string("%1.0f",owidth)+"."+string("%1.0f",owidth)+"s"
      cont=string(form,xr)
    endif
    cont=cont+" | "+string("%10.0f",r)
    cont=cont+string("%9.3f",r./sum(r))
    cont=cont+string("%11.3f",cumsum(r)./sum(r))
    freq=freq|cont
    freq=freq|ostring2+"-----------------------------------"
    cont=ostring3+" | "+string("%10.0f",sum(r))
    cont=cont+string("%9.3f",1.0)
;    cont=cont+string("%11.3f",1.0)
    freq=freq|cont
    freq=freq|ostring1+"==================================="
    freq=freq|" "
  endo
;
endp

