; -----------------------------------------------------------------------
; Library      wavelet
; -----------------------------------------------------------------------
; See_also     fwt invfwt fwt2 invfwt2 
; -----------------------------------------------------------------------
; Link         ../tutorials/wavestart.html Wavelet tutorial
; -----------------------------------------------------------------------
; Macro        fwttin
; -----------------------------------------------------------------------
; Description  Generates the translation invariant estimate of x with
;              automatic hardthresholding.
;              It is well-known that nonlinear wavelet estimators are not 
;              translation-invariant: if we shift the underlying data set 
;              by a small amount, apply nonlinear thresholding and shift 
;              the estimator back, then we usually obtain an estimator
;              different from the estimator without the shifting and
;              backshifting operation.
;              To get rid of this, we average over several estimators 
;              obtained by shifting, nonlinear thresholding and
;              backshifting.
; -----------------------------------------------------------------------
; Usage        xs = fwttin (x, l, h)
; Input
;   Parameter  x 
;   Definition n x 1 array, input data, observed at an equispaced grid
;   Parameter  l 
;   Definition p x 1 array, number of father wavelets   
;   Parameter  h
;   Definition q x 1 array, wavelet basis coefficients   
; Output
;   Parameter  xs
;   Definition n x 1 array, translation invariant estimator   
; -----------------------------------------------------------------------
; Example      ; loads the wavelet library
;              library ("wavelet")
;              ; generates an equidistant grid from 0 to 1 with 256 points
;              t  = grid (0, 2*pi./255, 256)
;              ; computes a sine function with a normal distributed error
;              x  = 3.*sin(t)+normal(256)
;              ; computes the translation invariant estimator
;              xi = fwttin (x, 4, haar)
;              ; creates a display for plotting
;              d  = createdisplay (1, 1)
;              ; plots the original data t~x and the estimator t~xi
;              show (d, 1, 1, t~x, t~xi)
; -----------------------------------------------------------------------
; Result       xs hardthresholded translation invariant estimator
; -----------------------------------------------------------------------
; Author       Sigbert Klinke, 970127, Sperlich 980105
; -----------------------------------------------------------------------
proc(xs)=fwttin(x,l,h)
  i = 0
  n = rows(x)
  xs = 0.*matrix(n)
  {ab, bb} = fwt (x, l, h)
  ba  = max(bb[,1])
  bt  = paf(bb[,3], bb[,1].=ba)
  vn  = 2.*sum(bt^2)/n
  ht  = sqrt(2.*vn.*log(n))
  bb1 = bb[,1:2]~(bb[,3].*(abs(bb[,3]).>abs(ht)))
  xs  = invfwt (ab, bb1, n, l, h)
  k   = 4*floor(log(n)/log(2))
  while (i<k)
    i = i+1
    xb = x[i+1:n]|x[1:i]
    {ab, bb} = fwt (xb, l, h)
    bb1 = bb[,1:2]~(bb[,3].*(abs(bb[,3]).>abs(ht)))
    xb  = invfwt (ab, bb1, n, l, h)
    xb  = xb[(n-i+1):n]|xb[1:(n-i)]
    xs  = xs + xb
  endo
  xs = xs / (k+1)
endp

